/*
 * Copyright 2012-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.boot.actuate.autoconfigure.web.server;

import java.util.function.Consumer;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

import org.springframework.boot.actuate.autoconfigure.endpoint.EndpointAutoConfiguration;
import org.springframework.boot.actuate.autoconfigure.endpoint.web.WebEndpointAutoConfiguration;
import org.springframework.boot.actuate.autoconfigure.web.servlet.ServletManagementContextAutoConfiguration;
import org.springframework.boot.autoconfigure.AutoConfigurations;
import org.springframework.boot.autoconfigure.web.servlet.DispatcherServletAutoConfiguration;
import org.springframework.boot.autoconfigure.web.servlet.ServletWebServerFactoryAutoConfiguration;
import org.springframework.boot.test.context.runner.WebApplicationContextRunner;
import org.springframework.boot.test.system.CapturedOutput;
import org.springframework.boot.test.system.OutputCaptureExtension;
import org.springframework.boot.web.servlet.context.AnnotationConfigServletWebServerApplicationContext;
import org.springframework.util.StringUtils;

import static org.assertj.core.api.Assertions.assertThat;

/**
 * Tests for {@link ManagementContextAutoConfiguration}.
 *
 * @author Madhura Bhave
 * @author Andy Wilkinson
 */
@ExtendWith(OutputCaptureExtension.class)
class ManagementContextAutoConfigurationTests {

	@Test
	void childManagementContextShouldStartForEmbeddedServer(CapturedOutput output) {
		WebApplicationContextRunner contextRunner = new WebApplicationContextRunner(
				AnnotationConfigServletWebServerApplicationContext::new)
			.withConfiguration(AutoConfigurations.of(ManagementContextAutoConfiguration.class,
					ServletWebServerFactoryAutoConfiguration.class, ServletManagementContextAutoConfiguration.class,
					WebEndpointAutoConfiguration.class, EndpointAutoConfiguration.class));
		contextRunner.withPropertyValues("server.port=0", "management.server.port=0")
			.run((context) -> assertThat(output).satisfies(numberOfOccurrences("Tomcat started on port", 2)));
	}

	@Test
	void childManagementContextShouldNotStartWithoutEmbeddedServer(CapturedOutput output) {
		WebApplicationContextRunner contextRunner = new WebApplicationContextRunner()
			.withConfiguration(AutoConfigurations.of(ManagementContextAutoConfiguration.class,
					ServletWebServerFactoryAutoConfiguration.class, ServletManagementContextAutoConfiguration.class,
					WebEndpointAutoConfiguration.class, EndpointAutoConfiguration.class));
		contextRunner.withPropertyValues("server.port=0", "management.server.port=0").run((context) -> {
			assertThat(context).hasNotFailed();
			assertThat(output).doesNotContain("Tomcat started");
		});
	}

	@Test
	void childManagementContextShouldRestartWhenParentIsStoppedThenStarted(CapturedOutput output) {
		WebApplicationContextRunner contextRunner = new WebApplicationContextRunner(
				AnnotationConfigServletWebServerApplicationContext::new)
			.withConfiguration(AutoConfigurations.of(ManagementContextAutoConfiguration.class,
					ServletWebServerFactoryAutoConfiguration.class, ServletManagementContextAutoConfiguration.class,
					WebEndpointAutoConfiguration.class, EndpointAutoConfiguration.class));
		contextRunner.withPropertyValues("server.port=0", "management.server.port=0").run((context) -> {
			assertThat(output).satisfies(numberOfOccurrences("Tomcat started on port", 2));
			context.getSourceApplicationContext().stop();
			context.getSourceApplicationContext().start();
			assertThat(output).satisfies(numberOfOccurrences("Tomcat started on port", 4));
		});
	}

	@Test
	void givenSamePortManagementServerWhenManagementServerAddressIsConfiguredThenContextRefreshFails() {
		WebApplicationContextRunner contextRunner = new WebApplicationContextRunner(
				AnnotationConfigServletWebServerApplicationContext::new)
			.withConfiguration(AutoConfigurations.of(ManagementContextAutoConfiguration.class,
					ServletWebServerFactoryAutoConfiguration.class, ServletManagementContextAutoConfiguration.class,
					WebEndpointAutoConfiguration.class, EndpointAutoConfiguration.class,
					DispatcherServletAutoConfiguration.class));
		contextRunner.withPropertyValues("server.port=0", "management.server.address=127.0.0.1")
			.run((context) -> assertThat(context).getFailure()
				.hasMessageStartingWith("Management-specific server address cannot be configured"));
	}

	private <T extends CharSequence> Consumer<T> numberOfOccurrences(String substring, int expectedCount) {
		return (charSequence) -> {
			int count = StringUtils.countOccurrencesOf(charSequence.toString(), substring);
			assertThat(count).isEqualTo(expectedCount);
		};
	}

}
