/*
 * Copyright 2012-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.boot.autoconfigure.web.servlet;

import java.net.URI;

import org.junit.jupiter.api.Test;

import org.springframework.boot.autoconfigure.context.PropertyPlaceholderAutoConfiguration;
import org.springframework.boot.autoconfigure.http.HttpMessageConvertersAutoConfiguration;
import org.springframework.boot.autoconfigure.thymeleaf.ThymeleafAutoConfiguration;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.web.client.TestRestTemplate;
import org.springframework.boot.test.web.server.LocalServerPort;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;

import static org.assertj.core.api.Assertions.assertThat;

/**
 * Integration tests for the welcome page.
 *
 * @author Madhura Bhave
 */
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT,
		properties = { "spring.web.resources.chain.strategy.content.enabled=true",
				"spring.thymeleaf.prefix=classpath:/templates/thymeleaf/" })
class WelcomePageIntegrationTests {

	@LocalServerPort
	private int port;

	private final TestRestTemplate template = new TestRestTemplate();

	@Test
	void contentStrategyWithWelcomePage() throws Exception {
		RequestEntity<?> entity = RequestEntity.get(new URI("http://localhost:" + this.port + "/"))
			.header("Accept", MediaType.ALL.toString())
			.build();
		ResponseEntity<String> content = this.template.exchange(entity, String.class);
		assertThat(content.getBody()).contains("/custom-");
		assertThat(content.getStatusCode()).isEqualTo(HttpStatus.OK);
	}

	@Test
	void notAcceptableWelcomePage() throws Exception {
		RequestEntity<?> entity = RequestEntity.get(new URI("http://localhost:" + this.port + "/"))
			.header("Accept", "spring/boot")
			.build();
		ResponseEntity<String> content = this.template.exchange(entity, String.class);
		assertThat(content.getStatusCode()).isEqualTo(HttpStatus.NOT_ACCEPTABLE);
	}

	@Configuration
	@Import({ PropertyPlaceholderAutoConfiguration.class, WebMvcAutoConfiguration.class,
			HttpMessageConvertersAutoConfiguration.class, ServletWebServerFactoryAutoConfiguration.class,
			DispatcherServletAutoConfiguration.class, ThymeleafAutoConfiguration.class })
	static class TestConfiguration {

		static void main(String[] args) {
			new SpringApplicationBuilder(TestConfiguration.class).run(args);
		}

	}

}
