/*
 * Copyright 2012-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.boot.test.autoconfigure.json.app;

import java.io.IOException;
import java.util.Date;
import java.util.UUID;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;

import org.springframework.boot.jackson.JsonComponent;
import org.springframework.boot.jackson.JsonObjectDeserializer;
import org.springframework.boot.jackson.JsonObjectSerializer;
import org.springframework.boot.test.autoconfigure.json.JsonTest;

/**
 * Example {@link JsonComponent @JsonComponent} for use with {@link JsonTest @JsonTest}
 * tests.
 *
 * @author Phillip Webb
 */
@JsonComponent
public class ExampleJsonComponent {

	static class Serializer extends JsonObjectSerializer<ExampleCustomObject> {

		@Override
		protected void serializeObject(ExampleCustomObject value, JsonGenerator jgen, SerializerProvider provider)
				throws IOException {
			jgen.writeStringField("value", value.value());
			jgen.writeNumberField("date", value.date().getTime());
			jgen.writeStringField("uuid", value.uuid().toString());
		}

	}

	static class Deserializer extends JsonObjectDeserializer<ExampleCustomObject> {

		@Override
		protected ExampleCustomObject deserializeObject(JsonParser jsonParser, DeserializationContext context,
				ObjectCodec codec, JsonNode tree) throws IOException {
			String value = nullSafeValue(tree.get("value"), String.class);
			Date date = nullSafeValue(tree.get("date"), Integer.class, Date::new);
			UUID uuid = nullSafeValue(tree.get("uuid"), String.class, UUID::fromString);
			return new ExampleCustomObject(value, date, uuid);
		}

	}

}
