// Copyright (C) 2016 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.gerrit.server;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.gerrit.server.notedb.ReviewerStateInternal.CC;
import static com.google.gerrit.server.notedb.ReviewerStateInternal.REVIEWER;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableTable;
import com.google.common.collect.Table;
import com.google.gerrit.entities.Account;
import com.google.gerrit.entities.PatchSetApproval;
import com.google.gerrit.server.notedb.ReviewerStateInternal;
import java.time.Instant;

/**
 * Set of reviewers on a change.
 *
 * <p>A given account may appear in multiple states and at different timestamps. No reviewers with
 * state {@link ReviewerStateInternal#REMOVED} are ever exposed by this interface.
 */
public class ReviewerSet {
  private static final ReviewerSet EMPTY = new ReviewerSet(ImmutableTable.of());

  public static ReviewerSet fromApprovals(Iterable<PatchSetApproval> approvals) {
    PatchSetApproval first = null;
    Table<ReviewerStateInternal, Account.Id, Instant> reviewers = HashBasedTable.create();
    for (PatchSetApproval psa : approvals) {
      if (first == null) {
        first = psa;
      } else {
        checkArgument(
            first.key().patchSetId().changeId().equals(psa.key().patchSetId().changeId()),
            "multiple change IDs: %s, %s",
            first.key(),
            psa.key());
      }
      Account.Id id = psa.accountId();
      reviewers.put(REVIEWER, id, psa.granted());
      if (psa.value() != 0) {
        reviewers.remove(CC, id);
      }
    }
    return new ReviewerSet(reviewers);
  }

  public static ReviewerSet fromTable(Table<ReviewerStateInternal, Account.Id, Instant> table) {
    return new ReviewerSet(table);
  }

  public static ReviewerSet empty() {
    return EMPTY;
  }

  private final ImmutableTable<ReviewerStateInternal, Account.Id, Instant> table;
  private ImmutableSet<Account.Id> accounts;

  private ReviewerSet(Table<ReviewerStateInternal, Account.Id, Instant> table) {
    this.table = ImmutableTable.copyOf(table);
  }

  public ImmutableSet<Account.Id> all() {
    if (accounts == null) {
      // Idempotent and immutable, don't bother locking.
      accounts = ImmutableSet.copyOf(table.columnKeySet());
    }
    return accounts;
  }

  public ImmutableSet<Account.Id> byState(ReviewerStateInternal state) {
    return table.row(state).keySet();
  }

  public ImmutableTable<ReviewerStateInternal, Account.Id, Instant> asTable() {
    return table;
  }

  @Override
  public boolean equals(Object o) {
    return (o instanceof ReviewerSet) && table.equals(((ReviewerSet) o).table);
  }

  @Override
  public int hashCode() {
    return table.hashCode();
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + table;
  }
}
