// Copyright (C) 2012 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.gerrit.server.git.validators;

import com.google.common.collect.ImmutableList;
import com.google.gerrit.server.validators.ValidationException;
import java.util.List;

/**
 * Exception thrown when a Git commit fails validations. Gerrit supports a wide range of validations
 * (for example it validates any commits pushed to NoteDb refs for format compliance or allows to
 * enforce commit message lengths to not exceed a certain length).
 */
public class CommitValidationException extends ValidationException {
  private static final long serialVersionUID = 1L;
  private final ImmutableList<CommitValidationMessage> messages;

  public CommitValidationException(String reason, CommitValidationMessage message) {
    super(reason);
    this.messages = ImmutableList.of(message);
  }

  public CommitValidationException(String reason, CommitValidationMessage message, Throwable why) {
    super(reason, why);
    this.messages = ImmutableList.of(message);
  }

  public CommitValidationException(String reason, List<CommitValidationMessage> messages) {
    super(reason);
    this.messages = ImmutableList.copyOf(messages);
  }

  public CommitValidationException(String reason) {
    super(reason);
    this.messages = ImmutableList.of();
  }

  public CommitValidationException(String reason, Throwable why) {
    super(reason, why);
    this.messages = ImmutableList.of();
  }

  /** Returns all validation messages individually. */
  public ImmutableList<CommitValidationMessage> getMessages() {
    return messages;
  }

  /** Returns all validation as a single, formatted string. */
  public String getFullMessage() {
    StringBuilder sb = new StringBuilder(getMessage());
    if (!messages.isEmpty()) {
      sb.append(':');
      for (CommitValidationMessage msg : messages) {
        sb.append("\n  ").append(msg.getMessage());
      }
    }
    return sb.toString();
  }
}
