/**
 * @license
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import * as path from "path";
import * as fs from "fs";
import { fail } from "./utils";

/** SharedLicensesProvider uses to get license's text
 * when package itself doesn't have LICENSE file.
 */
export class SharedLicensesProvider {
  private licenseNameToText: Map<string, string>;

  /** constructor takes list of file paths with licenses texts.
   * The path is used to load a license text,
   * the filename (without dirname) is used as a license name.
   * */
  public constructor(sharedLicensesFiles: string[]) {
    this.licenseNameToText = new Map();
    sharedLicensesFiles.forEach(f => {
      const licenseName = path.basename(f);
      if(this.licenseNameToText.has(licenseName)) {
        fail(`There are multiple files for the license's text '${licenseName}'`);
      }
      this.licenseNameToText.set(licenseName, fs.readFileSync(f, {encoding: 'utf-8'}));
    });
  }

  public getText(licenseName: string): string {
    if(!this.licenseNameToText.has(licenseName)) {
      fail(`There are no text for license ${licenseName}`);
    }
    return this.licenseNameToText.get(licenseName)!;
  }

}
