/**
 * @license
 * Copyright 2017 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import {assert} from '@open-wc/testing';
import {HookApi, PluginElement} from '../../../api/hook';
import {PluginApi} from '../../../api/plugin';
import '../../../test/common-test-setup';
import '../../shared/gr-js-api-interface/gr-js-api-interface';
import {GrDomHook, GrDomHooksManager} from './gr-dom-hooks';

suite('gr-dom-hooks tests', () => {
  let instance: GrDomHooksManager;
  let hook: HookApi<PluginElement>;
  let plugin: PluginApi;

  setup(() => {
    window.Gerrit.install(
      p => {
        plugin = p;
      },
      '0.1',
      'http://test.com/plugins/testplugin/static/test.js'
    );
    instance = new GrDomHooksManager(plugin);
  });

  suite('placeholder', () => {
    let createPlaceHolderStub: sinon.SinonStub;

    setup(() => {
      createPlaceHolderStub = sinon.stub(
        GrDomHook.prototype,
        '_createPlaceholder'
      );
      hook = instance.getDomHook('foo-bar');
    });

    test('registers placeholder class', () => {
      assert.isTrue(
        createPlaceHolderStub.calledWithExactly(
          'testplugin-autogenerated-foo-bar'
        )
      );
    });

    test('getModuleName()', () => {
      assert.equal(hook.getModuleName(), 'testplugin-autogenerated-foo-bar');
    });
  });

  suite('custom element', () => {
    setup(() => {
      hook = instance.getDomHook('foo-bar', 'my-el');
    });

    test('getModuleName()', () => {
      assert.equal(hook.getModuleName(), 'my-el');
    });

    test('onAttached', () => {
      const onAttachedSpy = sinon.spy();
      hook.onAttached(onAttachedSpy);
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      hook.handleInstanceAttached(el1);
      hook.handleInstanceAttached(el2);
      assert.isTrue(onAttachedSpy.firstCall.calledWithExactly(el1));
      assert.isTrue(onAttachedSpy.secondCall.calledWithExactly(el2));
    });

    test('onDetached', () => {
      const onDetachedSpy = sinon.spy();
      hook.onDetached(onDetachedSpy);
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      hook.handleInstanceDetached(el1);
      assert.isTrue(onDetachedSpy.firstCall.calledWithExactly(el1));
      hook.handleInstanceDetached(el2);
      assert.isTrue(onDetachedSpy.secondCall.calledWithExactly(el2));
    });

    test('getAllAttached', () => {
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      el1.textContent = 'one';
      el2.textContent = 'two';
      hook.handleInstanceAttached(el1);
      hook.handleInstanceAttached(el2);
      assert.deepEqual([el1, el2], hook.getAllAttached());
      hook.handleInstanceDetached(el1);
      assert.deepEqual([el2], hook.getAllAttached());
    });

    test('getLastAttached', () => {
      const beforeAttachedPromise = hook
        .getLastAttached()
        .then((el: HTMLElement) => assert.strictEqual(el1, el));
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      el1.textContent = 'one';
      el2.textContent = 'two';
      hook.handleInstanceAttached(el1);
      hook.handleInstanceAttached(el2);
      const afterAttachedPromise = hook
        .getLastAttached()
        .then((el: HTMLElement) => assert.strictEqual(el2, el));
      return Promise.all([beforeAttachedPromise, afterAttachedPromise]);
    });
  });
});
