/*!
Copyright (c) REBUILD <https://getrebuild.com/> and/or its owners. All rights reserved.

rebuild is dual-licensed under commercial and open source licenses (GPLv3).
See LICENSE and COMMERCIAL in the project root for license information.
*/

package com.rebuild.core.metadata.impl;

import cn.devezhao.commons.ObjectUtils;
import cn.devezhao.commons.ThrowableUtils;
import cn.devezhao.persist4j.Entity;
import cn.devezhao.persist4j.Field;
import cn.devezhao.persist4j.Record;
import cn.devezhao.persist4j.dialect.Dialect;
import cn.devezhao.persist4j.dialect.FieldType;
import cn.devezhao.persist4j.engine.ID;
import cn.devezhao.persist4j.metadata.CascadeModel;
import cn.devezhao.persist4j.metadata.impl.AnyEntity;
import cn.devezhao.persist4j.metadata.impl.FieldImpl;
import cn.devezhao.persist4j.util.StringHelper;
import cn.devezhao.persist4j.util.support.Table;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.hankcs.hanlp.HanLP;
import com.rebuild.core.Application;
import com.rebuild.core.metadata.EntityHelper;
import com.rebuild.core.metadata.MetadataHelper;
import com.rebuild.core.metadata.easymeta.DisplayType;
import com.rebuild.core.metadata.easymeta.EasyField;
import com.rebuild.core.metadata.easymeta.EasyMetaFactory;
import com.rebuild.core.privileges.UserHelper;
import com.rebuild.core.support.SetUser;
import com.rebuild.core.support.i18n.Language;
import com.rebuild.core.support.setup.Installer;
import com.rebuild.utils.BlockList;
import com.rebuild.utils.CommonsUtils;
import com.rebuild.utils.RbAssert;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.CharSet;
import org.apache.commons.lang.StringUtils;

import java.sql.DataTruncation;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import static com.rebuild.core.metadata.impl.EasyFieldConfigProps.DATETIME_FORMAT;
import static com.rebuild.core.metadata.impl.EasyFieldConfigProps.DATE_FORMAT;
import static com.rebuild.core.metadata.impl.EasyFieldConfigProps.NUMBER_CALCFORMULA;
import static com.rebuild.core.metadata.impl.EasyFieldConfigProps.NUMBER_CALCFORMULABACKEND;
import static com.rebuild.core.metadata.impl.EasyFieldConfigProps.NUMBER_NOTNEGATIVE;


@Slf4j
public class Field2Schema extends SetUser {

    
    protected static final int DDL_TIMEOUT = 60 * 15;

    
    private static final int DECIMAL_SCALE = 8;

    final protected Set<ID> recordedMetaIds = new HashSet<>();

    public Field2Schema() {
        super();
    }

    public Field2Schema(ID user) {
        super.setUser(user);
    }

    @Override
    public ID getUser() {
        ID user = super.getUser();
        RbAssert.isAllow(UserHelper.isSuperAdmin(user), Language.L("仅超级管理员可操作"));
        return user;
    }

    
    public String createField(Entity entity, String fieldLabel, DisplayType type, String comments, String refEntity, JSON extConfig) {
        return createField(entity, fieldLabel, null, type, comments, refEntity, extConfig);
    }

    
    public String createField(Entity entity, String fieldLabel, String fieldName, DisplayType type, String comments, String refEntity, JSON extConfig) {
        fieldLabel = CommonsUtils.maxstr(StringUtils.trim(fieldLabel), 40);
        fieldName = StringUtils.trim(fieldName);

        if (StringUtils.length(fieldName) < 4) fieldName = toPinyinName(fieldLabel);
        else fieldName = toPinyinName(fieldName);

        for (int i = 0; i < 6; i++) {
            if (entity.containsField(fieldName) || MetadataHelper.isCommonsField(fieldName)) {
                fieldName += CommonsUtils.randomInt(1, 99);
            } else {
                break;
            }
        }

        Field field = createUnsafeField(
                entity, fieldName, fieldLabel, type, true, true, true, true, true, comments, refEntity, null, extConfig, null);

        Collection<String> uniqueKeyFields = null;
        if (type == DisplayType.SERIES) uniqueKeyFields = Collections.singletonList(field.getName());

        boolean schemaReady = schema2Database(entity, new Field[]{field}, uniqueKeyFields);
        if (!schemaReady) {
            Application.getCommonsService().delete(recordedMetaIds.toArray(new ID[0]));
            throw new MetadataModificationException(Language.L("无法同步元数据到数据库"));
        }

        MetadataHelper.getMetadataFactory().refresh();
        return fieldName;
    }

    
    public boolean dropField(Field field, boolean force) {
        EasyField easyMeta = EasyMetaFactory.valueOf(field);
        ID metaRecordId = easyMeta.getMetaId();
        if (easyMeta.isBuiltin() || metaRecordId == null) {
            throw new MetadataModificationException(Language.L("系统内置，不允许删除"));
        }

        Entity entity = field.getOwnEntity();
        if (entity.getNameField().equals(field)) {
            throw new MetadataModificationException(Language.L("名称字段不允许删除"));
        }

        if (!force) {
            long count;
            if ((count = checkRecordCount(entity)) > 100000) {
                throw new MetadataModificationException(Language.L("实体记录过多 (%d)，增加/删除字段可能导致表损坏", count));
            }
        }

        String ddl = String.format("alter table `%s` drop column `%s`", entity.getPhysicalName(), field.getPhysicalName());
        try {
            Application.getSqlExecutor().execute(ddl, DDL_TIMEOUT);
        } catch (Throwable ex) {
            
            if (ThrowableUtils.getRootCause(ex).getLocalizedMessage().contains("exists")) {
                log.warn("Column not exists? " + ex.getLocalizedMessage());
            } else {
                log.error("DDL ERROR : \n" + ddl, ex);
                return false;
            }
        }

        Application.getBean(MetaFieldService.class).delete(metaRecordId);
        MetadataHelper.getMetadataFactory().refresh();
        return true;
    }

    
    protected long checkRecordCount(Entity entity) {
        String sql = String.format("select count(%s) from %s", entity.getPrimaryField().getName(), entity.getName());
        Object[] count = Application.createQueryNoFilter(sql).unique();
        return ObjectUtils.toLong(count[0]);
    }

    
    public boolean schema2Database(Entity entity, Field[] fields) {
        return schema2Database(entity, fields, null);
    }

    
    public boolean schema2Database(Entity entity, Field[] fields, Collection<String> uniqueKeyFields) {
        Dialect dialect = Application.getPersistManagerFactory().getDialect();
        final Table table = new Table(entity, dialect);
        final String alterSql = "alter table `" + entity.getPhysicalName() + "`";

        
        if (Installer.isUseH2() && fields.length > 1) {
            for (Field field : fields) {
                StringBuilder ddl = new StringBuilder(alterSql).append(" add column ");
                table.generateFieldDDL(field, ddl);

                try {
                    Application.getSqlExecutor().executeBatch(new String[]{ddl.toString()}, DDL_TIMEOUT);
                } catch (Throwable ex) {
                    log.error("DDL ERROR : \n" + ddl, ex);
                    return false;
                }
            }

            return true;
        }

        StringBuilder ddl = new StringBuilder(alterSql);
        for (Field field : fields) {
            ddl.append("\n  add column ");
            table.generateFieldDDL(field, ddl);
            ddl.append(",");

            
            if (uniqueKeyFields != null && uniqueKeyFields.contains(field.getName())) {
                
                
                ddl.append(MessageFormat.format("\n  add unique index UIX999_{0} ({0}),", field.getPhysicalName()));
            }
        }
        ddl.deleteCharAt(ddl.length() - 1);

        try {
            Application.getSqlExecutor().executeBatch(new String[]{ddl.toString()}, DDL_TIMEOUT);
        } catch (Throwable ex) {
            
            if (fields.length == 1
                    && ThrowableUtils.getRootCause(ex).getLocalizedMessage().contains("Duplicate column")) {
                log.warn("Duplicate column exists? " + ex.getLocalizedMessage());
            } else {
                log.error("DDL ERROR : \n" + ddl, ex);
                return false;
            }
        }
        return true;
    }

    
    public Field createUnsafeField(Entity entity, String fieldName, String fieldLabel, DisplayType dt,
                                   boolean nullable, boolean creatable, boolean updatable, boolean repeatable, boolean queryable, String comments, String refEntity, CascadeModel cascade,
                                   JSON extConfig, Object defaultValue) {
        if (dt == DisplayType.SERIES || EntityHelper.AutoId.equalsIgnoreCase(fieldName)) {
            nullable = false;
            creatable = false;
            updatable = false;
            repeatable = false;
        } else if (dt == DisplayType.BARCODE) {
            nullable = true;
            creatable = false;
            updatable = false;
            queryable = true;
        }else if (EntityHelper.QuickCode.equalsIgnoreCase(fieldName)) {
            creatable = false;
            queryable = false;
        }

        String physicalName = StringHelper.hyphenate(fieldName).toUpperCase();

        Record recordOfField = EntityHelper.forNew(EntityHelper.MetaField, getUser());
        recordOfField.setString("belongEntity", entity.getName());
        recordOfField.setString("fieldName", fieldName);
        recordOfField.setString("physicalName", physicalName);
        recordOfField.setString("fieldLabel", fieldLabel);
        recordOfField.setString("displayType", dt.name());
        recordOfField.setBoolean("nullable", nullable);
        recordOfField.setBoolean("creatable", creatable);
        recordOfField.setBoolean("updatable", updatable);
        recordOfField.setBoolean("repeatable", repeatable);
        recordOfField.setBoolean("queryable", queryable);

        if (StringUtils.isNotBlank(comments)) {
            recordOfField.setString("comments", comments);
        }
        if (defaultValue != null) {
            recordOfField.setString("defaultValue", defaultValue.toString());
        }

        if (dt == DisplayType.PICKLIST) {
            refEntity = "PickList";
        } else if (dt == DisplayType.CLASSIFICATION) {
            refEntity = "ClassificationData";
        } else if (dt == DisplayType.ANYREFERENCE) {
            refEntity = AnyEntity.FLAG;
        }

        if (extConfig != null) {
            recordOfField.setString("extConfig", extConfig.toJSONString());
        }

        if (StringUtils.isNotBlank(refEntity)) {
            
            
            if (!DynamicMetadataContextHolder.isSkipRefentityCheck(false)) {
                if (!(MetadataHelper.containsEntity(refEntity) || AnyEntity.FLAG.equals(refEntity))) {
                    throw new MetadataModificationException(Language.L("无效引用实体 : %s", refEntity));
                }
            }

            recordOfField.setString("refEntity", refEntity);
            if (cascade != null) {
                String cascadeAlias = cascade == CascadeModel.RemoveLinks ? "remove-links" : cascade.name().toLowerCase();
                recordOfField.setString("cascade", cascadeAlias);
            } else {
                recordOfField.setString("cascade", CascadeModel.Ignore.name().toLowerCase());
            }
        }

        int maxLength = dt.getMaxLength();
        if (EntityHelper.QuickCode.equalsIgnoreCase(fieldName)) {
            maxLength = 70;
        }
        recordOfField.setInt("maxLength", maxLength);

        if ((dt == DisplayType.REFERENCE || dt == DisplayType.N2NREFERENCE) && StringUtils.isBlank(refEntity)) {
            throw new MetadataModificationException(Language.L("引用字段必须指定引用实体"));
        }

        recordOfField = Application.getCommonsService().create(recordOfField);
        recordedMetaIds.add(recordOfField.getPrimary());

        

        boolean autoValue = EntityHelper.AutoId.equalsIgnoreCase(fieldName);

        
        nullable = !MetadataHelper.isCommonsField(fieldName)
                || (MetadataHelper.isApprovalField(fieldName) || fieldName.equalsIgnoreCase(EntityHelper.QuickCode));

        
        if (!EntityHelper.ApprovalState.equalsIgnoreCase(fieldName)) {
            defaultValue = null;
        }

        Field unsafeField = new FieldImpl(fieldName, physicalName, fieldLabel, null,
                creatable, updatable, Boolean.TRUE, entity, dt.getFieldType(), maxLength, CascadeModel.Ignore,
                nullable, repeatable, autoValue, DECIMAL_SCALE, defaultValue);
        if (entity instanceof UnsafeEntity) {
            ((UnsafeEntity) entity).addField(unsafeField);
        }
        return unsafeField;
    }

    
    protected String toPinyinName(final String text) {
        String identifier = text;

        
        if (identifier.length() >= 4 && identifier.matches("[a-zA-Z0-9_]+")) {
            if (!CharSet.ASCII_ALPHA.contains(identifier.charAt(0)) || BlockList.isBlock(identifier)) {
                identifier = "rb" + identifier;
            }
            return CommonsUtils.maxstr(identifier, 40);
        }

        identifier = HanLP.convertToPinyinString(identifier, "", false);
        identifier = identifier.replaceAll("[^a-zA-Z0-9]", "");
        if (StringUtils.isBlank(identifier)) {
            identifier = "rb" + CommonsUtils.randomInt(1000, 9999);
        }

        if (!CharSet.ASCII_ALPHA.contains(identifier.charAt(0))) {
            identifier = "rb" + identifier;
        }

        identifier = identifier.toLowerCase();
        if (identifier.length() > 40) {
            identifier = identifier.substring(0, 40);
        } else if (identifier.length() < 4) {
            identifier += CommonsUtils.randomInt(1000, 9999);
        }

        return identifier;
    }

    
    public boolean castType(Field field, DisplayType toType, boolean force) {
        EasyField fieldEasy = EasyMetaFactory.valueOf(field);
        ID metaRecordId = fieldEasy.getMetaId();
        if (fieldEasy.isBuiltin() || metaRecordId == null) {
            throw new MetadataModificationException(Language.L("系统内置，不允许转换"));
        }

        if (!force) {
            long count;
            if ((count = checkRecordCount(field.getOwnEntity())) > 100000) {
                throw new MetadataModificationException(Language.L("实体记录过多 (%d)，转换字段可能导致表损坏", count));
            }
        }

        Record fieldMeta = EntityHelper.forUpdate(metaRecordId, getUser(), false);
        fieldMeta.setString("displayType", toType.name());
        
        if (toType.getMaxLength() != FieldType.NO_NEED_LENGTH) {
            fieldMeta.setInt("maxLength", toType.getMaxLength());
        }
        
        JSONObject extraAttrs = fieldEasy.getExtraAttrs();
        if (!extraAttrs.isEmpty()) {
            extraAttrs.remove(DATE_FORMAT);
            extraAttrs.remove(DATETIME_FORMAT);
            Object notNegative = extraAttrs.remove(NUMBER_NOTNEGATIVE);
            Object calcFormula = extraAttrs.remove(NUMBER_CALCFORMULA);
            Object calcFormulaBackend = extraAttrs.remove(NUMBER_CALCFORMULABACKEND);

            extraAttrs.clear();
            if (notNegative != null) extraAttrs.put(NUMBER_NOTNEGATIVE, notNegative);
            if (calcFormula != null) extraAttrs.put(NUMBER_CALCFORMULA, calcFormula);
            if (calcFormulaBackend instanceof Boolean && (Boolean) calcFormulaBackend) extraAttrs.put(NUMBER_CALCFORMULABACKEND, true);

            if (extraAttrs.isEmpty()) fieldMeta.setNull("extConfig");
            else fieldMeta.setString("extConfig", extraAttrs.toJSONString());
        }
        Application.getCommonsService().update(fieldMeta, false);

        
        DynamicMetadataContextHolder.setSkipLanguageRefresh();
        MetadataHelper.getMetadataFactory().refresh();
        field = MetadataHelper.getField(field.getOwnEntity().getName(), field.getName());

        
        try {
            java.lang.reflect.Field defaultValueOfField = field.getClass().getDeclaredField("defaultValue");
            defaultValueOfField.setAccessible(true);
            defaultValueOfField.set(field, null);
            java.lang.reflect.Field nullableOfField = field.getClass().getDeclaredField("nullable");
            nullableOfField.setAccessible(true);
            nullableOfField.set(field, true);
        } catch (ReflectiveOperationException ignored) {
        }

        String alterTypeSql = null;
        try {
            Dialect dialect = Application.getPersistManagerFactory().getDialect();
            final Table table = new Table(field.getOwnEntity(), dialect);
            StringBuilder ddl = new StringBuilder();
            table.generateFieldDDL(field, ddl);

            alterTypeSql = String.format("alter table `%s` change column `%s` ",
                    field.getOwnEntity().getPhysicalName(), field.getPhysicalName());
            alterTypeSql += ddl.toString().trim().replace("  ", " ");

            Application.getSqlExecutor().executeBatch(new String[]{alterTypeSql}, DDL_TIMEOUT);
            log.info("Cast field type : {}", alterTypeSql);

        } catch (Throwable ex) {
            
            fieldMeta.setString("displayType", EasyMetaFactory.getDisplayType(field).name());
            Application.getCommonsService().update(fieldMeta, false);

            log.error("DDL ERROR : \n{}", alterTypeSql, ex);

            Throwable cause = ThrowableUtils.getRootCause(ex);
            String causeMsg = cause.getLocalizedMessage();
            if (cause instanceof DataTruncation) {
                causeMsg = Language.L("已有数据内容长度超出限制，无法完成转换");
            }
            throw new MetadataModificationException(causeMsg);

        } finally {
            MetadataHelper.getMetadataFactory().refresh();
            DynamicMetadataContextHolder.isSkipLanguageRefresh(true);
        }

        return true;
    }
}
