/*!
Copyright (c) REBUILD <https://getrebuild.com/> and/or its owners. All rights reserved.

rebuild is dual-licensed under commercial and open source licenses (GPLv3).
See LICENSE and COMMERCIAL in the project root for license information.
*/

package com.rebuild.core.privileges;

import cn.devezhao.bizz.security.member.BusinessUnit;
import cn.devezhao.bizz.security.member.Member;
import cn.devezhao.bizz.security.member.NoMemberFoundException;
import cn.devezhao.bizz.security.member.Role;
import cn.devezhao.bizz.security.member.Team;
import cn.devezhao.persist4j.Entity;
import cn.devezhao.persist4j.Record;
import cn.devezhao.persist4j.engine.ID;
import com.alibaba.fastjson.JSONArray;
import com.rebuild.core.Application;
import com.rebuild.core.RebuildException;
import com.rebuild.core.metadata.EntityHelper;
import com.rebuild.core.metadata.MetadataHelper;
import com.rebuild.core.privileges.bizz.CombinedRole;
import com.rebuild.core.privileges.bizz.Department;
import com.rebuild.core.privileges.bizz.User;
import com.rebuild.core.service.approval.FlowNode;
import com.rebuild.core.support.RebuildConfiguration;
import com.rebuild.utils.ImageMaker;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;

import java.io.File;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;


@Slf4j
public class UserHelper {

    
    public static final String DEFAULT_AVATAR = "/assets/img/avatar.png";

    
    public static boolean isAdmin(ID userId) {
        try {
            return Application.getUserStore().getUser(userId).isAdmin();
        } catch (NoMemberFoundException ex) {
            log.error("No User found : " + userId);
        }
        return false;
    }

    
    public static boolean isSuperAdmin(ID userId) {
        return UserService.ADMIN_USER.equals(userId) || UserService.SYSTEM_USER.equals(userId);
    }

    
    public static boolean isActive(ID bizzId) {
        try {
            switch (bizzId.getEntityCode()) {
                case EntityHelper.User:
                    return Application.getUserStore().getUser(bizzId).isActive();
                case EntityHelper.Department:
                    return !Application.getUserStore().getDepartment(bizzId).isDisabled();
                case EntityHelper.Role:
                    return !Application.getUserStore().getRole(bizzId).isDisabled();
                case EntityHelper.Team:
                    return !Application.getUserStore().getTeam(bizzId).isDisabled();
            }

        } catch (NoMemberFoundException ex) {
            log.error("No bizz found : " + bizzId);
        }
        return false;
    }

    
    public static Department getDepartment(ID userId) {
        try {
            User u = Application.getUserStore().getUser(userId);
            return u.getOwningDept();
        } catch (NoMemberFoundException ex) {
            log.error("No User found : " + userId);
        }
        return null;
    }

    
    public static Set<ID> getAllChildren(Department parent) {
        Set<ID> children = new HashSet<>();
        children.add((ID) parent.getIdentity());
        for (BusinessUnit child : parent.getAllChildren()) {
            children.add((ID) child.getIdentity());
        }
        return children;
    }

    
    public static String getName(ID bizzId) {
        try {
            switch (bizzId.getEntityCode()) {
                case EntityHelper.User:
                    return Application.getUserStore().getUser(bizzId).getFullName();
                case EntityHelper.Department:
                    return Application.getUserStore().getDepartment(bizzId).getName();
                case EntityHelper.Role:
                    return Application.getUserStore().getRole(bizzId).getName();
                case EntityHelper.Team:
                    return Application.getUserStore().getTeam(bizzId).getName();
            }

        } catch (NoMemberFoundException ex) {
            log.error("No bizz found : " + bizzId);
        }
        return null;
    }

    
    public static Member[] getMembers(ID groupId) {
        Set<Principal> ms = null;
        try {
            switch (groupId.getEntityCode()) {
                case EntityHelper.Department:
                    ms = Application.getUserStore().getDepartment(groupId).getMembers();
                    break;
                case EntityHelper.Role:
                    ms = Application.getUserStore().getRole(groupId).getMembers();
                    break;
                case EntityHelper.Team:
                    ms = Application.getUserStore().getTeam(groupId).getMembers();
                    break;
                default:
                    break;
            }

        } catch (NoMemberFoundException ex) {
            log.error("No group found : " + groupId);
        }

        if (ms == null || ms.isEmpty()) {
            return new Member[0];
        }
        
        return ms.toArray(new Member[0]);
    }

    
    public static Set<ID> parseUsers(JSONArray userDefs, ID record) {
        return parseUsers(userDefs, record, false);
    }

    
    public static Set<ID> parseUsers(JSONArray userDefs, ID record, boolean filterDisabled) {
        if (userDefs == null) return Collections.emptySet();

        Set<String> users = new HashSet<>();
        for (Object u : userDefs) {
            users.add((String) u);
        }
        return parseUsers(users, record, filterDisabled);
    }

    
    public static Set<ID> parseUsers(Collection<String> userDefs, ID record) {
        return parseUsers(userDefs, record, false);
    }

    
    public static Set<ID> parseUsers(Collection<String> userDefs, ID recordId, boolean filterDisabled) {
        Entity entity = recordId == null ? null : MetadataHelper.getEntity(recordId.getEntityCode());

        Set<ID> bizzs = new HashSet<>();
        Set<String> useFields = new HashSet<>();
        for (String def : userDefs) {
            if (ID.isId(def)) {
                bizzs.add(ID.valueOf(def));
            } else if (entity != null && MetadataHelper.getLastJoinField(entity, def) != null) {
                useFields.add(def);
            } else {
                if (FlowNode.USER_OWNS.equals(def));  
                else log.warn("Invalid field or id : {}", def);
            }
        }

        if (!useFields.isEmpty()) {
            useFields.add(entity.getPrimaryField().getName());
            Record bizzValue = Application.getQueryFactory().recordNoFilter(recordId, useFields.toArray(new String[0]));

            if (bizzValue != null) {
                for (String field : bizzValue.getAvailableFields()) {
                    Object value = bizzValue.getObjectValue(field);
                    if (value == null) continue;

                    if (value instanceof ID[]) {
                        CollectionUtils.addAll(bizzs, (ID[]) value);
                    } else {
                        bizzs.add((ID) value);
                    }
                }
            }
        }

        Set<ID> users = new HashSet<>();
        for (ID bizz : bizzs) {
            if (bizz.getEntityCode() == EntityHelper.User) {
                users.add(bizz);
            } else {
                Member[] ms = getMembers(bizz);
                for (Member m : ms) {
                    if (m.getIdentity().equals(UserService.SYSTEM_USER)) continue;
                    users.add((ID) m.getIdentity());
                }
            }
        }

        
        if (filterDisabled) {
            for (Iterator<ID> iter = users.iterator(); iter.hasNext(); ) {
                User u = Application.getUserStore().getUser(iter.next());
                if (!u.isActive()) iter.remove();
            }
        }

        return users;
    }

    
    public static File generateAvatar(String name, boolean forceMake) {
        if (StringUtils.isBlank(name)) name = "RB";

        File avatarFile = RebuildConfiguration.getFileOfData("avatar-" + name + "29.jpg");
        if (avatarFile.exists()) {
            if (forceMake) {
                FileUtils.deleteQuietly(avatarFile);
            } else {
                return avatarFile;
            }
        }

        ImageMaker.makeAvatar(name, avatarFile);
        return avatarFile;
    }

    
    public static ID findUserByFullName(String fullName) {
        for (User u : Application.getUserStore().getAllUsers()) {
            if (fullName.equalsIgnoreCase(u.getFullName())) {
                return u.getId();
            }
        }
        return null;
    }

    
    public static User[] sortUsers() {
        return sortUsers(Boolean.FALSE);
    }

    
    public static User[] sortUsers(boolean isAll) {
        User[] users = Application.getUserStore().getAllUsers();
        
        if (!isAll) {
            List<User> list = new ArrayList<>();
            for (User u : users) {
                if (u.isActive()) {
                    list.add(u);
                }
            }
            users = list.toArray(new User[0]);
        }

        sortMembers(users);
        return users;
    }

    
    public static Member[] sortMembers(Member[] members) {
        if (members == null || members.length == 0) {
            return new Member[0];
        }

        if (members[0] instanceof User) {
            Arrays.sort(members, Comparator.comparing(o -> ((User) o).getFullName()));
        } else {
            Arrays.sort(members, Comparator.comparing(Member::getName));
        }
        return members;
    }

    
    public static Set<ID> getRoleAppends(ID user) {
        Role role = Application.getUserStore().getUser(user).getOwningRole();
        if (role instanceof CombinedRole) {
            return ((CombinedRole) role).getRoleAppends();
        }
        return null;
    }

    
    public static Set<ID> getRolesOfUser(ID userId) {
        Role role = Application.getUserStore().getUser(userId).getOwningRole();
        Set<ID> s = new HashSet<>();
        if (role != null) {
            s.add((ID) role.getIdentity());
            if (role instanceof CombinedRole) {
                s.addAll(((CombinedRole) role).getRoleAppends());
            }
        }
        return Collections.unmodifiableSet(s);
    }

    
    public static Set<ID> getTeamsOfUser(ID userId) {
        Set<ID> s = new HashSet<>();
        for (Team t : Application.getUserStore().getUser(userId).getOwningTeams()) {
            s.add((ID) t.getIdentity());
        }
        return Collections.unmodifiableSet(s);
    }

    
    public static Set<ID> getMembersOfRole(ID roleId) {
        Object[][] array = Application.createQueryNoFilter(
                "select userId from RoleMember where roleId = ?")
                .setParameter(1, roleId)
                .array();

        Set<ID> s = new HashSet<>();
        for (Object[] o : array) s.add((ID) o[0]);
        return s;
    }

    
    public static boolean isSelf(ID user, ID otherUserOrAnyRecordId) {
        ID createdBy = otherUserOrAnyRecordId;
        if (otherUserOrAnyRecordId.getEntityCode() != EntityHelper.User) {
            createdBy = getCreatedBy(otherUserOrAnyRecordId);
            if (createdBy == null) return false;
        }

        if (createdBy.equals(user)) return true;

        
        return isAdmin(createdBy) && isAdmin(user);
    }

    private static ID getCreatedBy(ID anyRecordId) {
        final String ckey = "CreatedBy-" + anyRecordId;
        ID createdBy = (ID) Application.getCommonsCache().getx(ckey);
        if (createdBy != null) {
            return createdBy;
        }

        Entity entity = MetadataHelper.getEntity(anyRecordId.getEntityCode());
        if (!entity.containsField(EntityHelper.CreatedBy)) {
            log.warn("No [createdBy] field in [{}]", entity.getEntityCode());
            return null;
        }

        Object[] c = Application.getQueryFactory().uniqueNoFilter(anyRecordId, EntityHelper.CreatedBy);
        if (c == null) {
            throw new RebuildException("No record found : " + anyRecordId);
        }

        createdBy = (ID) c[0];
        Application.getCommonsCache().putx(ckey, createdBy);
        return createdBy;
    }
}
