/*!
Copyright (c) REBUILD <https://getrebuild.com/> and/or its owners. All rights reserved.

rebuild is dual-licensed under commercial and open source licenses (GPLv3).
See LICENSE and COMMERCIAL in the project root for license information.
*/

package com.rebuild.core.service.dashboard.charts;

import cn.devezhao.commons.ObjectUtils;
import cn.devezhao.persist4j.Entity;
import cn.devezhao.persist4j.Field;
import cn.devezhao.persist4j.Query;
import cn.devezhao.persist4j.dialect.FieldType;
import cn.devezhao.persist4j.engine.ID;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.rebuild.core.Application;
import com.rebuild.core.DefinedException;
import com.rebuild.core.metadata.EntityHelper;
import com.rebuild.core.metadata.MetadataHelper;
import com.rebuild.core.metadata.easymeta.DisplayType;
import com.rebuild.core.metadata.easymeta.EasyField;
import com.rebuild.core.metadata.easymeta.EasyMetaFactory;
import com.rebuild.core.metadata.impl.EasyFieldConfigProps;
import com.rebuild.core.privileges.UserHelper;
import com.rebuild.core.privileges.UserService;
import com.rebuild.core.service.query.AdvFilterParser;
import com.rebuild.core.service.query.ParseHelper;
import com.rebuild.core.support.SetUser;
import com.rebuild.core.support.general.FieldValueHelper;
import com.rebuild.core.support.i18n.Language;
import com.rebuild.utils.CommonsUtils;
import org.apache.commons.lang.StringUtils;

import java.text.DecimalFormat;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;


public abstract class ChartData extends SetUser implements ChartSpec {

    protected JSONObject config;

    private boolean fromPreview = false;
    
    private Map<String, Object> extraParams;

    
    protected ChartData(JSONObject config) {
        this.config = config;
    }

    
    public Map<String, Object> getExtraParams() {
        return extraParams == null ? Collections.emptyMap() : extraParams;
    }

    
    public ChartData setExtraParams(Map<String, Object> extraParams) {
        this.extraParams = extraParams;
        return this;
    }

    
    protected boolean isFromPreview() {
        return fromPreview;
    }

    
    public Entity getSourceEntity() {
        String e = config.getString("entity");
        return MetadataHelper.getEntity(e);
    }

    
    public String getTitle() {
        return StringUtils.defaultIfBlank(config.getString("title"), "未命名图表");
    }

    
    public Dimension[] getDimensions() {
        JSONArray items = config.getJSONObject("axis").getJSONArray("dimension");
        if (items == null || items.isEmpty()) {
            return new Dimension[0];
        }

        List<Dimension> list = new ArrayList<>();
        for (Object o : items) {
            JSONObject item = (JSONObject) o;
            Field[] validFields = getValidFields(item);
            Dimension dim = new Dimension(
                    validFields[0], getFormatSort(item), getFormatCalc(item),
                    item.getString("label"),
                    validFields[1]);
            list.add(dim);
        }
        return list.toArray(new Dimension[0]);
    }

    
    public Numerical[] getNumericals() {
        JSONArray items = config.getJSONObject("axis").getJSONArray("numerical");
        if (items == null || items.isEmpty()) {
            return new Numerical[0];
        }

        List<Numerical> list = new ArrayList<>();
        for (Object o : items) {
            JSONObject item = (JSONObject) o;
            Field[] validFields = getValidFields(item);
            Numerical num = new Numerical(
                    validFields[0], getFormatSort(item), getFormatCalc(item),
                    item.getString("label"),
                    item.getInteger("scale"),
                    item.getJSONObject("filter"),
                    validFields[1]);
            list.add(num);
        }
        return list.toArray(new Numerical[0]);
    }

    
    private Field[] getValidFields(JSONObject item) {
        String fieldName = item.getString("field");
        if (MetadataHelper.getLastJoinField(getSourceEntity(), fieldName) == null) {
            throw new DefinedException(Language.L("字段 [%s] 已不存在，请调整图表配置", fieldName.toUpperCase()));
        }

        Field[] fields = new Field[2];
        String[] fieldNames = fieldName.split("\\.");

        if (fieldNames.length > 1) {
            fields[1] = getSourceEntity().getField(fieldNames[0]);
            fields[0] = fields[1].getReferenceEntity().getField(fieldNames[1]);
        } else {
            fields[0] = getSourceEntity().getField(fieldNames[0]);
        }
        return fields;
    }

    
    private FormatSort getFormatSort(JSONObject item) {
        if (StringUtils.isNotBlank(item.getString("sort"))) {
            return FormatSort.valueOf(item.getString("sort"));
        }
        return FormatSort.NONE;
    }

    
    private FormatCalc getFormatCalc(JSONObject item) {
        if (StringUtils.isNotBlank(item.getString("calc"))) {
            return FormatCalc.valueOf(item.getString("calc"));
        }
        return FormatCalc.NONE;
    }

    
    protected String getFilterSql(Numerical withNumericalFilter) {
        String filterSql = getFilterSql();
        if (withNumericalFilter != null && withNumericalFilter.getFilter() != null) {
            String filter = new AdvFilterParser(withNumericalFilter.getFilter()).toSqlWhere();
            if (filter != null) filterSql = String.format("((%s) and (%s))", filterSql, filter);
        }
        return filterSql;
    }

    
    protected String getFilterSql() {
        String previewFilter = StringUtils.EMPTY;
        
        if (isFromPreview() && getSourceEntity().containsField(EntityHelper.AutoId)) {
            String maxAidSql = String.format("select max(autoId) from %s", getSourceEntity().getName());
            Object[] o = Application.createQueryNoFilter(maxAidSql).unique();
            long maxAid = ObjectUtils.toLong(o[0]);
            if (maxAid > 5000) {
                previewFilter = String.format("(%s >= %d) and ", EntityHelper.AutoId, Math.max(maxAid - 2000, 0));
            }
        }

        JSONObject filterExpr = config.getJSONObject("filter");
        if (ParseHelper.validAdvFilter(filterExpr)) {
            AdvFilterParser filterParser = new AdvFilterParser(filterExpr);
            String sqlWhere = filterParser.toSqlWhere();
            if (sqlWhere != null) {
                sqlWhere = previewFilter + sqlWhere;
            }
            return StringUtils.defaultIfBlank(sqlWhere, "(1=1)");
        }

        return previewFilter + "(1=1)";
    }

    
    protected String getSortSql() {
        Set<String> sorts = new HashSet<>();
        for (Axis dim : getDimensions()) {
            FormatSort fs = dim.getFormatSort();
            if (fs != FormatSort.NONE) {
                sorts.add(dim.getSqlName() + " " + fs.toString().toLowerCase());
            }
        }





        for (Numerical num : getNumericals()) {
            FormatSort fs = num.getFormatSort();
            if (fs != FormatSort.NONE) {
                sorts.add(num.getSqlName() + " " + fs.toString().toLowerCase());
            }
        }
        return sorts.isEmpty() ? null : String.join(", ", sorts);
    }

    
    protected String wrapAxisValue(Numerical numerical, Object value) {
        return wrapAxisValue(numerical, value, Boolean.FALSE);
    }

    
    protected String wrapAxisValue(Numerical numerical, Object value, boolean useThousands) {
        if (ChartsHelper.isZero(value)) {
            return ChartsHelper.VALUE_ZERO;
        }

        String format = "###";
        if (numerical.getScale() > 0) {
            format = "##0.";
            format = StringUtils.rightPad(format, format.length() + numerical.getScale(), "0");
        }

        if (useThousands) format = "#," + format;

        if (ID.isId(value)) value = 1;

        String n = new DecimalFormat(format).format(value);
        if (useThousands) n = formatAxisValue(numerical, n);
        return n;
    }

    
    private String formatAxisValue(Numerical numerical, String value) {
        String type = getNumericalFlag(numerical);
        if (type == null) return value;

        if ("%".equals(type)) value += "%";
        else if (type.contains("%s")) value = String.format(type, value);
        else value = type + " " + value;
        return value;
    }

    
    protected String getNumericalFlag(Numerical numerical) {
        if (numerical.getField().getType() != FieldType.DECIMAL) return null;

        if (!(numerical.getFormatCalc() == FormatCalc.SUM
                || numerical.getFormatCalc() == FormatCalc.AVG
                || numerical.getFormatCalc() == FormatCalc.MIN
                || numerical.getFormatCalc() == FormatCalc.MAX)) {
            return null;
        }

        String type = EasyMetaFactory.valueOf(numerical.getField()).getExtraAttr(EasyFieldConfigProps.DECIMAL_TYPE);
        if (type == null || "0".equalsIgnoreCase(type)) return null;
        else return type;
    }

    
    protected String wrapAxisValue(Dimension dimension, Object value) {
        return wrapAxisValue(dimension, value, Boolean.FALSE);
    }

    
    protected String wrapAxisValue(Dimension dimension, Object value, boolean useRefLink) {
        if (value == null || value == ChartsHelper.VALUE_NONE) {
            return ChartsHelper.VALUE_NONE;
        }

        EasyField axisField = EasyMetaFactory.valueOf(dimension.getField());
        DisplayType axisType = axisField.getDisplayType();

        String label;
        if (axisType == DisplayType.REFERENCE
                || axisType == DisplayType.CLASSIFICATION
                || axisType == DisplayType.BOOL
                || axisType == DisplayType.PICKLIST
                || axisType == DisplayType.STATE) {
            label = (String) FieldValueHelper.wrapFieldValue(value, axisField, true);
            label = CommonsUtils.escapeHtml(label);

            if (useRefLink && axisType == DisplayType.REFERENCE
                    && ID.valueOf(value.toString()).getEntityCode() > 100) {
                label = String.format("<a href='/app/redirect?id=%s&type=newtab'>%s</a>", value, label);
            }

        } else {
            label = value.toString();
            label = CommonsUtils.escapeHtml(label);
        }
        return label;
    }

    
    public JSON build(boolean fromPreview) {
        this.fromPreview = fromPreview;
        try {
            return this.build();
        } finally {
            this.fromPreview = false;
        }
    }

    
    protected Query createQuery(String sql) {
        if (this.fromPreview) {
            return Application.createQuery(sql, this.getUser());
        }

        boolean noPrivileges = false;
        JSONObject option = config.getJSONObject("option");
        if (option != null) {
            noPrivileges = option.getBooleanValue("noPrivileges");
        }
        String co = config.getString("chartOwning");
        ID chartOwning = ID.isId(co) ? ID.valueOf(co) : null;

        if (chartOwning == null || !noPrivileges) {
            return Application.createQuery(sql, this.getUser());
        } else {
            
            return Application.createQuery(sql,
                    UserHelper.isAdmin(chartOwning) ? UserService.SYSTEM_USER : this.getUser());
        }
    }

    
    protected String buildSql(Dimension dim, Numerical[] nums, boolean withFilter) {
        List<String> numSqlItems = new ArrayList<>();
        for (Numerical num : nums) {
            numSqlItems.add(num.getSqlName());
        }

        String sql = "select {0},{1} from {2} where {3} group by {0}";
        sql = MessageFormat.format(sql,
                dim.getSqlName(),
                StringUtils.join(numSqlItems, ", "),
                getSourceEntity().getName(), getFilterSql(withFilter ? nums[0] : null));
        return appendSqlSort(sql);
    }

    
    protected String buildSql(Dimension[] dims, Numerical num) {
        List<String> dimSqlItems = new ArrayList<>();
        for (Dimension dim : dims) {
            dimSqlItems.add(dim.getSqlName());
        }

        String sql = "select {0},{1} from {2} where {3} group by {0}";
        sql = MessageFormat.format(sql,
                StringUtils.join(dimSqlItems, ", "),
                num.getSqlName(),
                getSourceEntity().getName(),
                getFilterSql());
        return appendSqlSort(sql);
    }

    
    protected String buildSql(Dimension dim, Numerical num, boolean withFilter) {
        String sql = "select {0},{1} from {2} where {3} group by {0}";
        String where = getFilterSql(withFilter ? num : null);

        sql = MessageFormat.format(sql,
                dim.getSqlName(),
                num.getSqlName(),
                getSourceEntity().getName(), where);
        return appendSqlSort(sql);
    }

    
    protected String buildSql(Numerical num, boolean withFilter) {
        String sql = "select {0} from {1} where {2}";
        String where = getFilterSql(withFilter ? num : null);

        sql = MessageFormat.format(sql, num.getSqlName(), getSourceEntity().getName(), where);
        return appendSqlSort(sql);
    }

    
    protected String appendSqlSort(String sql) {
        String sorts = getSortSql();
        if (sorts != null) sql += " order by " + sorts;
        return sql;
    }

    
    protected boolean hasNumericalFilter(Numerical[] nums) {
        for (Numerical num : nums) {
            if (num.getFilter() != null) return true;
        }
        return false;
    }

    
    protected Object[][] mergeAxisEntry2Data(List<AxisEntry> axisValues, int indexAndSize, boolean useComparison) {
        if (useComparison) {
            
            List<AxisEntry[]> merged = new ArrayList<>();
            for (int i = 0; i < indexAndSize; i++) {
                int irow = 0;
                for (AxisEntry e : axisValues) {
                    if (e.getIndex() == i) {
                        AxisEntry[] ee = null;
                        try {
                            ee = merged.get(irow++);
                        } catch (IndexOutOfBoundsException ignored){}
                        if (ee == null) {
                            ee = new AxisEntry[indexAndSize];
                            merged.add(ee);
                        }
                        ee[i] = e;
                    }
                }
            }

            
            List<Object[]> dataRawList = new ArrayList<>();
            String nullLang = Language.L("无");
            for (AxisEntry[] group : merged) {
                List<String> keyName = new ArrayList<>();
                for (AxisEntry e : group) {
                    if (e == null || e.getKeyRaw() == null || e.getKeyRaw()[0] == null) keyName.add(nullLang);
                    else keyName.add(e.getKeyRaw()[0].toString());
                }

                Object[] d = new Object[indexAndSize + 1];
                d[0] = StringUtils.join(keyName, " - ");
                for (int i = 0; i < group.length; i++) {
                    d[i + 1] = group[i] == null ? 0 : group[i].getValue();
                }
                dataRawList.add(d);
            }

            return dataRawList.toArray(new Object[0][]);
        }

        
        Map<String, AxisEntry[]> merged = new LinkedHashMap<>();
        for (AxisEntry e : axisValues) {
            AxisEntry[] eee = merged.computeIfAbsent(e.getKey(), k -> new AxisEntry[indexAndSize]);
            eee[e.getIndex()] = e;
        }

        
        int startIndex = getDimensions().length;
        List<Object[]> dataRawList = new ArrayList<>();
        for (AxisEntry[] group : merged.values()) {
            AxisEntry keyItem = group[0];
            for (AxisEntry item : group) {
                if (keyItem != null) break;
                keyItem = item;
            }

            Object[] data = keyItem.getKeyRaw();
            data = Arrays.copyOf(data, startIndex + indexAndSize);

            for (AxisEntry item : group) {
                if (item != null) {
                    data[startIndex + item.getIndex()] = item.getValue();
                }
            }
            dataRawList.add(data);
        }

        return dataRawList.toArray(new Object[0][]);
    }
}
