/*!
Copyright (c) REBUILD <https://getrebuild.com/> and/or its owners. All rights reserved.

rebuild is dual-licensed under commercial and open source licenses (GPLv3).
See LICENSE and COMMERCIAL in the project root for license information.
*/

package com.rebuild.core.service.trigger.impl;

import cn.devezhao.bizz.privileges.impl.BizzPermission;
import cn.devezhao.commons.CalendarUtils;
import cn.devezhao.commons.ObjectUtils;
import cn.devezhao.persist4j.Field;
import cn.devezhao.persist4j.Record;
import cn.devezhao.persist4j.dialect.FieldType;
import cn.devezhao.persist4j.engine.ID;
import cn.devezhao.persist4j.engine.StandardRecord;
import cn.devezhao.persist4j.metadata.MissingMetaExcetion;
import cn.devezhao.persist4j.record.RecordVisitor;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.rebuild.core.Application;
import com.rebuild.core.configuration.general.AutoFillinManager;
import com.rebuild.core.metadata.EntityHelper;
import com.rebuild.core.metadata.MetadataHelper;
import com.rebuild.core.metadata.easymeta.DisplayType;
import com.rebuild.core.metadata.easymeta.EasyDateTime;
import com.rebuild.core.metadata.easymeta.EasyField;
import com.rebuild.core.metadata.easymeta.EasyMetaFactory;
import com.rebuild.core.metadata.easymeta.MultiValue;
import com.rebuild.core.privileges.UserService;
import com.rebuild.core.privileges.bizz.InternalPermission;
import com.rebuild.core.service.general.GeneralEntityServiceContextHolder;
import com.rebuild.core.service.general.OperatingContext;
import com.rebuild.core.service.query.QueryHelper;
import com.rebuild.core.service.trigger.ActionContext;
import com.rebuild.core.service.trigger.ActionType;
import com.rebuild.core.service.trigger.RobotTriggerObserver;
import com.rebuild.core.service.trigger.TriggerException;
import com.rebuild.core.service.trigger.TriggerResult;
import com.rebuild.core.service.trigger.aviator.AviatorUtils;
import com.rebuild.core.support.general.ContentWithFieldVars;
import com.rebuild.core.support.general.N2NReferenceSupport;
import com.rebuild.core.support.state.StateHelper;
import com.rebuild.utils.CommonsUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;


@Slf4j
public class FieldWriteback extends FieldAggregation {

    private static final ReentrantLock LOCK = new ReentrantLock();

    private FieldWritebackRefresh fieldWritebackRefresh;

    
    public static final String ONE2ONE_MODE = "one2one";

    private static final String DATE_EXPR = "#";
    private static final String CODE_PREFIX = "{{{{";  

    protected Set<ID> targetRecordIds;
    protected Record targetRecordData;

    public FieldWriteback(ActionContext context) {
        super(context, Boolean.TRUE);
    }

    @Override
    public ActionType getType() {
        return ActionType.FIELDWRITEBACK;
    }

    @Override
    public void clean() {
        super.clean();

        if (fieldWritebackRefresh != null) {
            log.info("Clear after refresh : {}", fieldWritebackRefresh);
            fieldWritebackRefresh.refresh();
            fieldWritebackRefresh = null;
        }
    }

    @Override
    public Object execute(OperatingContext operatingContext) throws TriggerException {
        boolean lockMode38 = ((JSONObject) actionContext.getActionContent()).getBooleanValue("lockMode");
        if (lockMode38) {
            long s = System.currentTimeMillis();
            log.info("Lock resources for {}", actionContext.getConfigId());
            LOCK.lock();

            s = System.currentTimeMillis() - s;
            if (s > 1000) log.warn("Lock acquired use {}ms. {}", s, actionContext.getConfigId());

            try {
                return this.execute38(operatingContext);
            } finally {
                LOCK.unlock();
            }
        } else {
            return this.execute38(operatingContext);
        }
    }

    private Object execute38(OperatingContext operatingContext) throws TriggerException {
        final String chainName = String.format("%s:%s:%s", actionContext.getConfigId(),
                operatingContext.getFixedRecordId(), operatingContext.getAction().getName());
        final List<String> tschain = checkTriggerChain(chainName);
        if (tschain == null) return TriggerResult.triggerOnce();

        this.prepare(operatingContext);

        if (targetRecordIds.isEmpty()) {
            log.debug("No target record(s) found");
            return TriggerResult.noMatching();
        }

        if (targetRecordData.isEmpty()) {
            if (!RobotTriggerObserver._TriggerLessLog) log.info("No data of target record : {}", targetRecordIds);
            return TriggerResult.targetEmpty();
        }

        final boolean forceUpdate = ((JSONObject) actionContext.getActionContent()).getBooleanValue("forceUpdate");
        final boolean stopPropagation = ((JSONObject) actionContext.getActionContent()).getBooleanValue("stopPropagation");

        List<ID> affected = new ArrayList<>();
        boolean targetSame = false;

        for (ID targetRecordId : targetRecordIds) {
            
            if (operatingContext.getAction() == BizzPermission.DELETE
                    && targetRecordId.equals(operatingContext.getFixedRecordId())) {
                continue;
            }

            
            if (!QueryHelper.exists(targetRecordId)) {
                log.warn("Target record dose not exists: {} (On {})", targetRecordId, actionContext.getConfigId());
                continue;
            }

            Record targetRecord = targetRecordData.clone();
            targetRecord.setID(targetEntity.getPrimaryField().getName(), targetRecordId);
            targetRecord.setDate(EntityHelper.ModifiedOn, CalendarUtils.now());
            targetRecord.setID(EntityHelper.ModifiedBy, UserService.SYSTEM_USER);

            
            if (isCurrentSame(targetRecord)) {
                if (!RobotTriggerObserver._TriggerLessLog) log.info("Ignore execution because the record are same : {}", targetRecordId);
                targetSame = true;
                continue;
            }

            
            GeneralEntityServiceContextHolder.setSkipGuard(targetRecordId);

            
            if (forceUpdate) {
                GeneralEntityServiceContextHolder.setAllowForceUpdate(targetRecordId);
            }
            
            if (stopPropagation) {
                GeneralEntityServiceContextHolder.setQuickMode();
            }

            
            GeneralEntityServiceContextHolder.setRepeatedCheckMode(GeneralEntityServiceContextHolder.RCM_CHECK_MAIN);

            List<String> tschainCurrentLoop = new ArrayList<>(tschain);
            tschainCurrentLoop.add(chainName);
            TRIGGER_CHAIN.set(tschainCurrentLoop);
            if (CommonsUtils.DEVLOG) System.out.println("[dev] Use current-loop tschain : " + tschainCurrentLoop);

            try {
                Application.getBestService(targetEntity).createOrUpdate(targetRecord);
                affected.add(targetRecord.getPrimary());

            } finally {
                GeneralEntityServiceContextHolder.isSkipGuardOnce();
                if (forceUpdate) GeneralEntityServiceContextHolder.isAllowForceUpdateOnce();
                if (stopPropagation) GeneralEntityServiceContextHolder.isQuickMode(true);
                GeneralEntityServiceContextHolder.getRepeatedCheckModeOnce();
            }
        }

        if (targetSame && affected.isEmpty()) return TriggerResult.targetSame();
        else return TriggerResult.success(affected);
    }

    @Override
    public void prepare(OperatingContext operatingContext) throws TriggerException {
        if (targetRecordIds != null) return;

        
        String[] targetFieldEntity = ((JSONObject) actionContext.getActionContent()).getString("targetEntity").split("\\.");
        sourceEntity = actionContext.getSourceEntity();
        targetEntity = MetadataHelper.getEntity(targetFieldEntity[1]);

        
        boolean isOne2One = ((JSONObject) actionContext.getActionContent()).getBooleanValue(ONE2ONE_MODE);

        targetRecordIds = new HashSet<>();

        
        if (TARGET_ANY.equals(targetFieldEntity[0])) {
            TargetWithMatchFields targetWithMatchFields = new TargetWithMatchFields();
            ID[] ids = targetWithMatchFields.matchMultiple(actionContext);
            CollectionUtils.addAll(targetRecordIds, ids);
        }
        
        else if (SOURCE_SELF.equalsIgnoreCase(targetFieldEntity[0])) {
            targetRecordIds.add(actionContext.getSourceRecord());
        }
        
        else if (isOne2One) {
            Record afterRecord = operatingContext.getAfterRecord();
            if (afterRecord == null) return;

            if (afterRecord.hasValue(targetFieldEntity[0])) {
                Object o = afterRecord.getObjectValue(targetFieldEntity[0]);
                
                if (o instanceof ID[]) {
                    Collections.addAll(targetRecordIds, (ID[]) o);
                } else if (o instanceof ID) {
                    targetRecordIds.add((ID) o);
                }

                
                boolean clearFields = ((JSONObject) actionContext.getActionContent()).getBooleanValue("clearFields");
                if (clearFields) {
                    Record beforeRecord = operatingContext.getBeforeRecord();
                    Object beforeValue = beforeRecord == null ? null : beforeRecord.getObjectValue(targetFieldEntity[0]);
                    if (beforeValue != null && !beforeValue.equals(o)) {
                        fieldWritebackRefresh = new FieldWritebackRefresh(this, beforeValue);
                    }
                }

            } else {
                Object[] o = Application.getQueryFactory().uniqueNoFilter(afterRecord.getPrimary(),
                        targetFieldEntity[0], afterRecord.getEntity().getPrimaryField().getName());
                if (o != null && o[0] != null) {
                    
                    if (o[0] instanceof ID[]) {
                        Collections.addAll(targetRecordIds, (ID[]) o[0]);
                    } else {
                        targetRecordIds.add((ID) o[0]);
                    }
                }
            }
        }
        
        else {
            
            Field targetField = targetEntity.getField(targetFieldEntity[0]);
            if (targetField.getType() == FieldType.REFERENCE_LIST) {
                Set<ID> set = N2NReferenceSupport.findReferences(targetField, operatingContext.getFixedRecordId());
                targetRecordIds.addAll(set);
            } else {
                String sql = String.format("select %s from %s where %s = ?",
                        targetEntity.getPrimaryField().getName(), targetFieldEntity[1], targetFieldEntity[0]);
                Object[][] array = Application.createQueryNoFilter(sql)
                        .setParameter(1, operatingContext.getFixedRecordId())
                        .array();

                for (Object[] o : array) {
                    targetRecordIds.add((ID) o[0]);
                }
            }
        }

        if (targetRecordIds.isEmpty()) {
            log.debug("Target record(s) are empty.");
        } else {
            targetRecordData = buildTargetRecordData(operatingContext, false);
        }
    }

    
    protected Record buildTargetRecordData(OperatingContext operatingContext, Boolean fromRefresh) {
        
        final boolean clearFields = ((JSONObject) actionContext.getActionContent()).getBooleanValue("clearFields");
        final boolean forceVNull = fromRefresh || (clearFields && operatingContext.getAction() == InternalPermission.DELETE_BEFORE);

        final Record targetRecord = EntityHelper.forNew(targetEntity.getEntityCode(), UserService.SYSTEM_USER, false);
        final JSONArray items = ((JSONObject) actionContext.getActionContent()).getJSONArray("items");

        final Set<String> fieldVars = new HashSet<>();
        final Set<String> fieldVarsN2NPath = new HashSet<>();
        
        Record useSourceData = null;

        if (!forceVNull) {
            for (Object o : items) {
                JSONObject item = (JSONObject) o;
                String sourceField = item.getString("sourceField");
                String updateMode = item.getString("updateMode");
                
                if (updateMode == null) {
                    updateMode = sourceField.contains(DATE_EXPR) ? "FORMULA" : "FIELD";
                }

                if ("FIELD".equalsIgnoreCase(updateMode)) {
                    fieldVars.add(sourceField);
                } else if ("FORMULA".equalsIgnoreCase(updateMode)) {
                    if (sourceField.contains(DATE_EXPR) && !sourceField.startsWith(CODE_PREFIX)) {
                        fieldVars.add(sourceField.split(DATE_EXPR)[0]);
                    } else {
                        Set<String> matchsVars = ContentWithFieldVars.matchsVars(sourceField);
                        for (String field : matchsVars) {
                            if (N2NReferenceSupport.isN2NMixPath(field, sourceEntity)) {
                                fieldVarsN2NPath.add(field);
                            } else {
                                if (MetadataHelper.getLastJoinField(sourceEntity, field) == null) {
                                    throw new MissingMetaExcetion(field, sourceEntity.getName());
                                }
                                fieldVars.add(field);
                            }
                        }
                    }
                }
            }

            if (!fieldVars.isEmpty()) {
                String sql = MessageFormat.format("select {0},{1} from {2} where {1} = ?",
                        StringUtils.join(fieldVars, ","),
                        sourceEntity.getPrimaryField().getName(),
                        sourceEntity.getName());
                useSourceData = Application.createQueryNoFilter(sql).setParameter(1, actionContext.getSourceRecord()).record();
            }
            if (!fieldVarsN2NPath.isEmpty()) {
                if (useSourceData == null) useSourceData = new StandardRecord(sourceEntity, null);
                fieldVars.addAll(fieldVarsN2NPath);

                for (String field : fieldVarsN2NPath) {
                    Object[] n2nVal = N2NReferenceSupport.getN2NValueByMixPath(field, actionContext.getSourceRecord());
                    useSourceData.setObjectValue(field, n2nVal);
                }
            }
        }

        for (Object o : items) {
            JSONObject item = (JSONObject) o;
            String targetField = item.getString("targetField");
            if (!MetadataHelper.checkAndWarnField(targetEntity, targetField)) continue;

            EasyField targetFieldEasy = EasyMetaFactory.valueOf(targetEntity.getField(targetField));

            String updateMode = item.getString("updateMode");
            String sourceAny = item.getString("sourceField");

            
            if ("VNULL".equalsIgnoreCase(updateMode) || forceVNull) {
                targetRecord.setNull(targetField);
            }

            
            else if ("VFIXED".equalsIgnoreCase(updateMode)) {
                RecordVisitor.setValueByLiteral(targetField, sourceAny, targetRecord);
            }

            
            else if ("FIELD".equalsIgnoreCase(updateMode)) {
                Field sourceFieldMeta = MetadataHelper.getLastJoinField(sourceEntity, sourceAny);
                if (sourceFieldMeta == null) continue;

                Object value = Objects.requireNonNull(useSourceData).getObjectValue(sourceAny);
                Object newValue = value == null ? null
                        : EasyMetaFactory.valueOf(sourceFieldMeta).convertCompatibleValue(value, targetFieldEasy);
                if (newValue != null) {
                    targetRecord.setObjectValue(targetField, newValue);
                } else if (clearFields) {
                    targetRecord.setNull(targetField);
                }
            }

            
            else if ("FORMULA".equalsIgnoreCase(updateMode)) {
                if (useSourceData == null) {
                    log.warn("[useSourceData] is null, Set to empty");
                    useSourceData = new StandardRecord(sourceEntity, null);
                }

                
                final boolean useCode = sourceAny.startsWith(CODE_PREFIX);

                
                if (sourceAny.contains(DATE_EXPR) && !useCode) {
                    String fieldName = sourceAny.split(DATE_EXPR)[0];
                    Field sourceField2 = MetadataHelper.getLastJoinField(sourceEntity, fieldName);
                    if (sourceField2 == null) continue;

                    Object value = useSourceData.getObjectValue(fieldName);
                    Object newValue = value == null ? null
                            : ((EasyDateTime) EasyMetaFactory.valueOf(sourceField2)).convertCompatibleValue(value, targetFieldEasy, sourceAny);
                    if (newValue != null) {
                        targetRecord.setObjectValue(targetField, newValue);
                    } else if (clearFields) {
                        targetRecord.setNull(targetField);
                    }
                }

                
                
                else {
                    String clearFormula = useCode
                            ? sourceAny.substring(4, sourceAny.length() - 4)
                            : sourceAny
                                .replace("×", "*")
                                .replace("÷", "/")
                                .replace("`", "\"");  

                    Map<String, Object> envMap = new HashMap<>();

                    for (String fieldName : fieldVars) {
                        String replace = "{" + fieldName + "}";
                        String replaceWhitQuote = "\"" + replace + "\"";
                        String replaceWhitQuoteSingle = "'" + replace + "'";
                        boolean forceUseQuote = false;

                        if (clearFormula.contains(replaceWhitQuote)) {
                            clearFormula = clearFormula.replace(replaceWhitQuote, fieldName);
                            forceUseQuote = true;
                        } else if (clearFormula.contains(replaceWhitQuoteSingle)) {
                            clearFormula = clearFormula.replace(replaceWhitQuoteSingle, fieldName);
                            forceUseQuote = true;
                        } else if (clearFormula.contains(replace)) {
                            clearFormula = clearFormula.replace(replace, fieldName);
                        } else {
                            continue;
                        }

                        Object value = useSourceData.getObjectValue(fieldName);

                        
                        Field varField = MetadataHelper.getLastJoinField(sourceEntity, fieldName);
                        EasyField easyVarField = varField == null ? null : EasyMetaFactory.valueOf(varField);
                        boolean isMultiField = easyVarField != null && (easyVarField.getDisplayType() == DisplayType.MULTISELECT
                                || easyVarField.getDisplayType() == DisplayType.TAG || easyVarField.getDisplayType() == DisplayType.N2NREFERENCE);
                        
                        boolean isStateField = easyVarField != null && easyVarField.getDisplayType() == DisplayType.STATE;

                        if (isStateField) {
                            value = value == null ? "" : StateHelper.getLabel(varField, (Integer) value);
                        } else if (value instanceof Date) {
                            value = CalendarUtils.getUTCDateTimeFormat().format(value);
                        } else if (value == null) {
                            
                            Field isN2NField = sourceEntity.containsField(fieldName) ? sourceEntity.getField(fieldName) : null;
                            
                            if (varField != null
                                    && (varField.getType() == FieldType.LONG || varField.getType() == FieldType.DECIMAL)) {
                                value = 0L;
                            } else if (fieldVarsN2NPath.contains(fieldName)
                                    || (isN2NField != null && isN2NField.getType() == FieldType.REFERENCE_LIST)) {
                                
                            } else {
                                value = StringUtils.EMPTY;
                            }
                        } else if (isMultiField) {
                            
                            if (easyVarField.getDisplayType() == DisplayType.N2NREFERENCE
                                    && targetFieldEasy.getDisplayType() == DisplayType.N2NREFERENCE) {
                                value = StringUtils.join((ID[]) value, MultiValue.MV_SPLIT);
                            } else {
                                
                                EasyField fakeTextField = EasyMetaFactory.valueOf(MetadataHelper.getField("User", "fullName"));
                                value = easyVarField.convertCompatibleValue(value, fakeTextField);
                            }
                        } else if (value instanceof ID || forceUseQuote) {
                            value = value.toString();
                        }

                        
                        if (value instanceof Long) value = BigDecimal.valueOf((Long) value);

                        envMap.put(fieldName, value);
                    }

                    Object newValue = AviatorUtils.eval(clearFormula, envMap, Boolean.FALSE);

                    if (newValue != null) {
                        DisplayType targetType = targetFieldEasy.getDisplayType();
                        if (targetType == DisplayType.NUMBER) {
                            targetRecord.setLong(targetField, CommonsUtils.toLongHalfUp(newValue));
                        } else if (targetType == DisplayType.DECIMAL) {
                            targetRecord.setDouble(targetField, ObjectUtils.toDouble(newValue));
                        } else if (targetType == DisplayType.DATE || targetType == DisplayType.DATETIME) {
                            if (newValue instanceof Date) {
                                targetRecord.setDate(targetField, (Date) newValue);
                            } else {
                                Date newValueCast = CalendarUtils.parse(newValue.toString());
                                if (newValueCast == null) log.warn("Cannot cast string to date : {}", newValue);
                                else targetRecord.setDate(targetField, newValueCast);
                            }
                        } else {
                            newValue = checkoutFieldValue(newValue, targetFieldEasy);
                            if (newValue != null) {
                                targetRecord.setObjectValue(targetField, newValue);
                            } else if (clearFields) {
                                targetRecord.setNull(targetField);
                            }
                        }
                    } else if (clearFields) {
                        targetRecord.setNull(targetField);
                    }
                }
            }
        }

        return targetRecord;
    }

    
    private Object checkoutFieldValue(Object value, EasyField field) {
        DisplayType dt = field.getDisplayType();
        Object newValue = null;

        if (dt == DisplayType.PICKLIST || dt == DisplayType.CLASSIFICATION
                || dt == DisplayType.REFERENCE || dt == DisplayType.ANYREFERENCE) {

            ID id = ID.isId(value) ? ID.valueOf(value.toString()) : null;
            if (id != null) {
                int entityCode = id.getEntityCode();
                if (dt == DisplayType.PICKLIST) {
                    if (entityCode == EntityHelper.PickList) newValue = id;
                } else if (dt == DisplayType.CLASSIFICATION) {
                    if (entityCode == EntityHelper.ClassificationData) newValue = id;
                } else if (dt == DisplayType.REFERENCE) {
                    if (field.getRawMeta().getReferenceEntity().getEntityCode() == entityCode) newValue = id;
                } else {
                    newValue = id;
                }
            }

        } else if (dt == DisplayType.N2NREFERENCE) {

            
            Object[] ids;
            if (value instanceof String) ids = value.toString().split(",");
            else ids = CommonsUtils.toArray(value);

            Set<ID> idsSet = new LinkedHashSet<>();
            for (Object id : ids) {
                if (id instanceof ID) idsSet.add((ID) id);
                else {
                    id = id.toString().trim();
                    if (ID.isId(id)) idsSet.add(ID.valueOf(id.toString()));
                }
            }
            
            newValue = idsSet.toArray(new ID[0]);

        } else if (dt == DisplayType.BOOL) {

            if (value instanceof Boolean) {
                newValue = value;
            } else {
                newValue = BooleanUtils.toBooleanObject(value.toString());
            }

        } else if (dt == DisplayType.MULTISELECT || dt == DisplayType.STATE) {

            if (value instanceof Integer || value instanceof Long) {
                newValue = value;
            }

        } else {
            
            newValue = value.toString();
        }

        if (newValue == null) {
            log.warn("Value `{}` cannot be convert to field (value) : {}", value, field.getRawMeta());
        }
        return newValue;
    }
}
