/*!
Copyright (c) REBUILD <https://getrebuild.com/> and/or its owners. All rights reserved.

rebuild is dual-licensed under commercial and open source licenses (GPLv3).
See LICENSE and COMMERCIAL in the project root for license information.
*/

package com.rebuild.core.support.general;

import cn.devezhao.persist4j.Entity;
import cn.devezhao.persist4j.Field;
import cn.devezhao.persist4j.dialect.FieldType;
import cn.devezhao.persist4j.engine.ID;
import cn.devezhao.persist4j.query.compiler.SelectItem;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.rebuild.core.Application;
import com.rebuild.core.UserContextHolder;
import com.rebuild.core.configuration.ConfigBean;
import com.rebuild.core.configuration.general.ClassificationManager;
import com.rebuild.core.configuration.general.MultiSelectManager;
import com.rebuild.core.configuration.general.PickListManager;
import com.rebuild.core.metadata.easymeta.DisplayType;
import com.rebuild.core.metadata.easymeta.EasyField;
import com.rebuild.core.metadata.easymeta.EasyMetaFactory;
import com.rebuild.core.privileges.UserHelper;
import com.rebuild.core.privileges.bizz.ZeroEntry;
import com.rebuild.core.support.ConfigurationItem;
import com.rebuild.core.support.RebuildConfiguration;
import com.rebuild.utils.JSONUtils;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class DataListWrapper {

    final protected int total;
    final protected Object[][] data;
    final protected SelectItem[] selectFields;
    final protected Entity entity;

    
    protected ID user;
    protected Map<String, Integer> queryJoinFields;

    
    protected boolean useDesensitized = false;

    private boolean mixWrapper = true;

    private Map<ID, Object> cacheRefValue = new HashMap<>();

    
    public DataListWrapper(int total, Object[][] data, SelectItem[] selectFields, Entity entity) {
        this.total = total;
        this.data = data;
        this.selectFields = selectFields;
        this.entity = entity;
    }

    
    protected void setPrivilegesFilter(ID user, Map<String, Integer> joinFields) {
        if (user != null && joinFields != null && !joinFields.isEmpty()) {
            this.user = user;
            this.queryJoinFields = joinFields;
        }

        if (user != null) {
            this.useDesensitized = !Application.getPrivilegesManager().allow(user, ZeroEntry.AllowNoDesensitized);
            if (!this.useDesensitized) {
                this.useDesensitized = UserHelper.isAdmin(user) && RebuildConfiguration.getBool(ConfigurationItem.SecurityEnhanced);
            }
        }
    }

    
    public JSON toJson() {
        final Field nameFiled = entity.getNameField();
        final EasyField nameFieldEasy = EasyMetaFactory.valueOf(nameFiled);

        final int joinFieldsLen = queryJoinFields == null ? 0 : queryJoinFields.size();
        final int selectFieldsLen = selectFields.length - joinFieldsLen;

        for (int rowIndex = 0; rowIndex < data.length; rowIndex++) {
            final Object[] raw = data[rowIndex];

            Object[] row = raw;
            if (joinFieldsLen > 0) {
                row = new Object[selectFieldsLen];
                System.arraycopy(raw, 0, row, 0, selectFieldsLen);
                data[rowIndex] = row;
            }

            Object nameValue = null;
            for (int colIndex = 0; colIndex < selectFieldsLen; colIndex++) {
                if (!checkHasFieldPrivileges(selectFields[colIndex])) {
                    row[colIndex] = FieldValueHelper.NO_READ_PRIVILEGES;
                    continue;
                }
                if (!checkHasJoinFieldPrivileges(selectFields[colIndex], raw)) {
                    row[colIndex] = FieldValueHelper.NO_READ_PRIVILEGES;
                    continue;
                }

                final Object value = row[colIndex];
                if (value == null) {
                    row[colIndex] = StringUtils.EMPTY;
                    continue;
                }

                final SelectItem fieldItem = selectFields[colIndex];
                final Field fieldMeta = fieldItem.getField();

                
                if (fieldMeta.equals(nameFiled) && !fieldItem.getFieldPath().contains(".")) {
                    nameValue = value;
                }

                
                if (colIndex + 1 == selectFieldsLen && fieldMeta.getType() == FieldType.PRIMARY) {
                    
                    if (nameValue == null) {
                        nameValue = FieldValueHelper.getLabel((ID) value, StringUtils.EMPTY);
                    } else {
                        nameValue = FieldValueHelper.wrapFieldValue(nameValue, nameFiled, true);
                    }

                    if (nameValue != null && isUseDesensitized(nameFieldEasy)) {
                        nameValue = FieldValueHelper.desensitized(nameFieldEasy, nameValue);
                    }

                    ((ID) value).setLabel(ObjectUtils.defaultIfNull(nameValue, StringUtils.EMPTY));
                }

                row[colIndex] = wrapFieldValue(value, fieldMeta);
            }
        }

        return JSONUtils.toJSONObject(
                new String[] { "total", "data" },
                new Object[] { total, data });
    }

    
    protected Object wrapFieldValue(Object value, Field field) {
        EasyField easyField = EasyMetaFactory.valueOf(field);
        if (easyField.getDisplayType() == DisplayType.ID) {
            return FieldValueHelper.wrapMixValue((ID) value, null);
        }

        final DisplayType dt = easyField.getDisplayType();
        final Object originValue = value;
        final boolean isCacheRefValue = dt == DisplayType.REFERENCE && value instanceof ID;

        if (isCacheRefValue) {
            if (cacheRefValue.containsKey((ID) value)) return cacheRefValue.get(value);
        }

        boolean unpack = dt == DisplayType.CLASSIFICATION || dt == DisplayType.PICKLIST
                || dt == DisplayType.STATE || dt == DisplayType.BOOL;

        value = FieldValueHelper.wrapFieldValue(value, easyField, unpack);

        if (value != null) {
            if (isUseDesensitized(easyField)) {
                value = FieldValueHelper.desensitized(easyField, value);
            }

            
            if (this.useDesensitized
                    && (dt == DisplayType.REFERENCE || dt == DisplayType.N2NREFERENCE)) {
                Field useNameField = easyField.getRawMeta().getReferenceEntity().getNameField();
                EasyField useNameFieldEasy = EasyMetaFactory.valueOf(useNameField);
                if (useNameFieldEasy.isDesensitized()) {
                    FieldValueHelper.desensitizedMixValue(useNameFieldEasy, (JSON) value);
                }
            }
        }

        
        if (value != null && this.mixWrapper) {
            if (easyField.getDisplayType() == DisplayType.PICKLIST) {
                String color = PickListManager.instance.getColor((ID) originValue);
                if (StringUtils.isNotBlank(color)) {
                    value = JSONUtils.toJSONObject(
                            new String[]{ "text", "color" }, new Object[]{ value, color });
                }

            } else if (easyField.getDisplayType() == DisplayType.MULTISELECT) {
                

                List<Object> colorLabels = new ArrayList<>();
                ConfigBean[] entries = MultiSelectManager.instance.getPickListRaw(field, false);
                for (ConfigBean e : entries) {
                    long m = e.get("mask", Long.class);
                    if (((long) originValue & m) != 0) {
                        String text = e.getString("text");
                        String color = e.getString("color");

                        if (StringUtils.isBlank(color)) {
                            colorLabels.add(text);
                        } else {
                            colorLabels.add(JSONUtils.toJSONObject(
                                    new String[]{ "text", "color" }, new Object[]{ text, color }));
                        }
                    }
                }

                ((JSONObject) value).put("text", colorLabels);

            } else if (easyField.getDisplayType() == DisplayType.TAG) {

                Map<String, String> colorNames = TagSupport.getNamesColor(easyField);
                List<Object> colorValue = new ArrayList<>();
                for (Object o : (JSONArray) value) {
                    String name = o.toString();
                    colorValue.add(JSONUtils.toJSONObject(
                            new String[]{ "name", "color" }, new Object[]{ name, colorNames.get(name) }));
                }
                value = colorValue;

            } else  if (easyField.getDisplayType() == DisplayType.CLASSIFICATION) {
                String color = ClassificationManager.instance.getColor((ID) originValue);
                if (StringUtils.isNotBlank(color)) {
                    value = JSONUtils.toJSONObject(
                            new String[]{ "text", "color" }, new Object[]{ value, color });
                }
            }
        }

        if (isCacheRefValue) cacheRefValue.put((ID) originValue, value);
        return value;
    }

    
    private boolean isUseDesensitized(EasyField easyField) {
        return this.useDesensitized && easyField.isDesensitized();
    }

    
    protected boolean checkHasJoinFieldPrivileges(SelectItem field, Object[] original) {
        if (this.queryJoinFields == null || UserHelper.isAdmin(user)) {
            return true;
        }

        String[] fieldPath = field.getFieldPath().split("\\.");
        if (fieldPath.length == 1) {
            return true;
        }

        int fieldIndex = queryJoinFields.get(fieldPath[0]);
        Object check = original[fieldIndex];
        return check == null || Application.getPrivilegesManager().allowRead(user, (ID) check);
    }

    
    protected boolean checkHasFieldPrivileges(SelectItem field) {
        ID u = user == null ? UserContextHolder.getUser() : user;
        return Application.getPrivilegesManager().getFieldPrivileges().isReadble(field.getField(), u);
    }

    
    public void setMixWrapper(boolean mixWrapper) {
        this.mixWrapper = mixWrapper;
    }
}
