/*!
Copyright (c) REBUILD <https://getrebuild.com/> and/or its owners. All rights reserved.

rebuild is dual-licensed under commercial and open source licenses (GPLv3).
See LICENSE and COMMERCIAL in the project root for license information.
*/

package com.rebuild.core.support.setup;

import cn.devezhao.commons.CalendarUtils;
import com.rebuild.core.BootEnvironmentPostProcessor;
import com.rebuild.core.support.ConfigurationItem;
import com.rebuild.core.support.RebuildConfiguration;
import com.rebuild.utils.CommandUtils;
import com.rebuild.utils.CompressUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.SystemUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.io.IOException;


@Slf4j
public class DatabaseBackup {

    final private String[] ignoreTables;

    public DatabaseBackup() {
        this(null);
    }

    public DatabaseBackup(String[] ignoreTables) {
        this.ignoreTables = ignoreTables;
    }

    
    public File backup() throws IOException {
        File backupdir = RebuildConfiguration.getFileOfData("_backups");
        if (!backupdir.exists()) FileUtils.forceMkdir(backupdir);

        return backup(backupdir);
    }

    
    public File backup(File backups) throws IOException {
        String url = BootEnvironmentPostProcessor.getProperty("db.url");
        String user = BootEnvironmentPostProcessor.getProperty("db.user");
        String passwd = BootEnvironmentPostProcessor.getProperty("db.passwd");

        url = url.split("\\?")[0].split("//")[1];
        String host = url.split(":")[0];
        String port = url.split("/")[0].split(":")[1];
        String dbname = url.split("/")[1];

        String destName = "backup_database." + CalendarUtils.getPlainDateTimeFormat().format(CalendarUtils.now());
        File dest = new File(backups, destName);

        String mysqldump = RebuildConfiguration.get(ConfigurationItem.MysqldumpBin);
        if (StringUtils.isBlank(mysqldump)) mysqldump = SystemUtils.IS_OS_WINDOWS ? "mysqldump.exe" : "mysqldump";
        
        
        String cmd = String.format(
                "%s -u%s -p\"%s\" -h%s -P%s --default-character-set=utf8 --opt --extended-insert=true --triggers --hex-blob --single-transaction -R %s>\"%s\"",
                mysqldump, user, passwd, host, port, dbname, dest.getAbsolutePath());

        if (ignoreTables != null) {
            String igPrefix = " --ignore-table=" + dbname + ".";
            String ig = igPrefix + StringUtils.join(ignoreTables, igPrefix);
            cmd = cmd.replaceFirst(" -R ", " -R" + ig + " ");
        }

        String echo = CommandUtils.execFor(cmd, true);
        boolean isGotError = echo.contains("Got error");
        if (isGotError) throw new RuntimeException(echo);

        File zip = new File(backups, destName + ".zip");
        try {
            CompressUtils.forceZip(zip, dest, null);

            FileUtils.deleteQuietly(dest);
            dest = zip;
        } catch (Exception e) {
            log.warn("Cannot zip backup : {}", zip);
        }

        log.info("Backup succeeded : {} ({})", dest, FileUtils.byteCountToDisplaySize(dest.length()));

        
        
        

        return dest;
    }
}
