#pragma once
#include "../acl_cpp_define.hpp"
#include <vector>
#include <map>
#include "../stdlib/string.hpp"
#include "redis_command.hpp"

#if !defined(ACL_CLIENT_ONLY) && !defined(ACL_REDIS_DISABLE)

namespace acl
{

#define GEO_INVALID		360
#define GEO_LONGITUDE_MIN	-180
#define GEO_LONGITUDE_MAX	180
#define GEO_LATITUDE_MIN	-85.05112878
#define GEO_LATITUDE_MAX	85.05112878

enum
{
	GEO_UNIT_FT,
	GEO_UNIT_M,
	GEO_UNIT_MI,
	GEO_UNIT_KM,
};

enum
{
	GEO_WITH_COORD = 1 << 0,
	GEO_WITH_DIST  = 1 << 1,
	GEO_WITH_HASH  = 1 << 2,
};

enum
{
	GEO_SORT_NONE,
	GEO_SORT_ASC,
	GEO_SORT_DESC,
};

class ACL_CPP_API geo_member
{
public:
	geo_member(const char* name);
	geo_member(const geo_member& member);
	~geo_member(void);

	void set_name(const char* name);
	const char* get_name() const
	{
		return name_.c_str();
	}

	void set_dist(double dist);
	double get_dist() const
	{
		return dist_;
	}

#if defined(_WIN32) || defined(_WIN64)
	void set_hash(__int64 hash);
	__int64 get_hash() const
#else
	void set_hash(long long int hash);
	long long int get_hash() const
#endif // defined(_WIN32) || defined(_WIN64)
	{
		return hash_;
	}

	void set_coordinate(double longitude, double latitude);
	double get_longitude() const
	{
		return longitude_;
	}

	double get_latitude() const
	{
		return latitude_;
	}

private:
	string name_;
	double dist_;
#if defined(_WIN32) || defined(_WIN64)
	__int64 hash_;
#else
	long long int hash_;
#endif // defined(_WIN32) || defined(_WIN64)

	double longitude_;
	double latitude_;
};

class ACL_CPP_API redis_geo : virtual public redis_command
{
public:
	/**
	 * see redis_command::redis_command()
	 */
	redis_geo();

	/**
	 * see redis_command::redis_command(redis_client*)
	 */
	redis_geo(redis_client* conn);

	/**
	 * see redis_command::redis_command(redis_client_cluster*)
	 */
	redis_geo(redis_client_cluster* cluster);

	ACL_CPP_DEPRECATED
	redis_geo(redis_client_cluster* cluster, size_t max_conns);

	redis_geo(redis_client_pipeline* pipeline);

	virtual ~redis_geo();

	/////////////////////////////////////////////////////////////////////

	/**
	 * һָĵλָ key 
	 * Add the specified geospatial item (latitude, logitude, name)
	 * to the specified key.
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param member {const char*} õıʶ
	 *  the geospatial's identifier
	 * @param longitude {double} 
	 *  the geospatial's loginitude
	 * @param latitude {double} γ
	 *  the geospatial's latitude
	 * @return {int} 1ӳɹ0õʶѴڣʹֵ޸ģ
	 *  Ҳ 0-1ʾ
	 *  the return value as below:
	 *  1: add one new member successfully
	 *  0: the member already existed, and the geospatial may be changed
	 * -1: some erro happened
	 */
	int geoadd(const char* key, const char* member,
		double longitude, double latitude);

	/**
	 * ָ key һַλ
	 * Add the specified geospatial items (latitude, logitude, name)
	 * to the specified key.
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param size {size_t} ĳ
	 *  the array's size
	 * @param members {const char* []} Ա飬䳤 size ָ
	 *  the members array, which's length was specified by size parameter
	 * @param longitudes {const double[]} 飬䳤 size ָ
	 *  the logintitudes array, which's length was specifed by size parameter
	 * @param latitudes {const double[]} γ飬䳤 size ָ
	 *  the lattitudes array, which's length was specifed by size parameter
	 * @return {int} ӳɹĳԱֵ£
	 *  return the successfully added members's count:
	 *  > 0: ʾɹӵĳԱ
	 *       represent the successfully added members's count
	 *    0: ЩԱѾ
	 *       the members's belong the key already existing
	 *   -1: ʾͨ result_error 鿴ԭ
	 *       some error happened, the result_error function can be used
	 *       to find the error's reason
	 */
	int geoadd(const char* key, size_t size, const char* members[],
		const double longitudes[], const double latitudes[]);

	/**
	 * ָ key һַλ
	 * Add the specified geospatial items (latitude, logitude, name)
	 * to the specified key.
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param members {const std::vector<string>&} Ա
	 *  the members array
	 * @param longitudes {const std::vector<double>&} 
	 *  the logintitudes array
	 * @param latitudes {const std::vector<double>&} γ
	 *  the lattitudes array
	 * @return {int} ӳɹĳԱֵ£
	 *  return the successfully added members's count:
	 *  > 0: ʾɹӵĳԱ
	 *       represent the successfully added members's count
	 *    0: ЩԱѾ
	 *       the members's belong the key already existing
	 *   -1: ʾͨ result_error 鿴ԭ
	 *       some error happened, the result_error function can be used
	 *       to find the error's reason
	 *  ע⣺(members, longitudes, latitudes)鳤ȱ
	 *  Notice: the three array's length must be equal between members,
	 *    longitudes and latitudes
	 */
	int geoadd(const char* key, const std::vector<string>& members,
		const std::vector<double>& longitudes,
		const std::vector<double>& latitudes);

	/**
	 * ַʽָԱ GEOHASH ֵ
	 * Returns members of a geospatial index as standard geohash strings.
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param members {const std::vector<string>&} Ա
	 *  the members array
	 * @param results {std::vector<string>&} 洢
	 *  store the result
	 * @return {bool} Ƿɹ
	 *  if the operation was successful.
	 */
	bool geohash(const char* key, const std::vector<string>& members,
		std::vector<string>& results);

	/**
	 * ַʽָԱ GEOHASH ֵ
	 * Returns members of a geospatial index as standard geohash strings.
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param member {const char*} Ա
	 *  the member of a geospatial index
	 * @param result {std::vector<string>&} 洢
	 *  store the result
	 * @return {bool} Ƿɹ
	 *  if the operation was successful.
	 */
	bool geohash(const char* key, const char* member, string& result);

	/**
	 * ָԱĵλ
	 * Returns longitude and latitude of members of a geospatial index
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param members {const std::vector<string>&} Ա
	 *  the members array
	 * @param results {std::vector<std::pair<double, double> >&} 洢
	 *  store the results
	 * @return {bool} Ƿɹ
	 *  if the operation was successful. 
	 */
	bool geopos(const char* key, const std::vector<string>& members,
		std::vector<std::pair<double, double> >& results);

	/**
	 * ĳָԱĵλ
	 * Returns longitude and latitude of the one member of
	 * a geospatial index
	 * @param key {const char*} ֵָ
	 *  the specifed key
	 * @param member {const char*} ָԱ
	 *  the specified member
	 * @param result {std::pair<double, double>&} 洢
	 *  store the result of longitude and latitude of the member
	 * @return {bool} Ƿɹ
	 *  if the operation was successful.
	 */
	bool geopos(const char* key, const char* member,
		std::pair<double, double>& result);

	/**
	 * λ֮ľ
	 * Returns the distance between two members of a geospatial index
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param member1 {const char*} Ա
	 *  one member of a geospatial index
	 * @param member2 {const char*} Ա
	 *  another member of a geospatial index
	 * @param unit {int} صľĵλֵ
	 * @return {double} ֮ĳȣֵ < 0 ʾ
	 *  returns the distance between two members, which was less than 0
	 *  if some error happened.
	 */
	double geodist(const char* key, const char* member1,
		const char* member2, int unit = GEO_UNIT_M);

	/**
	 * þĳָλڸ뷶Χڵ
	 * Query a sorted set representing a geospatial index to fetch
	 * members matching a given maximum distance from a point
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param longitude {double} ָľֵ
	 *  the longitude of the specified geospatial coordinate
	 * @param latitude {double} ָγֵ
	 *  the latitude of the specified geospatial coordinate
	 * @param radius {double} ޶ľ뷶ΧС
	 *  the distance from the specified coordinate
	 * @param unit {int} radius ĵλ
	 *  the unit type of the raidus
	 * @param with {int} ѯѡμĶ壺GEO_WITH_XXX
	 *  the serach operations, defined as GEO_WITH_XXX above
	 * @param sort {int} ѯʽμGEO_SORT_XXX
	 *  the sorted type of the results, defined as GEO_SORT_XXX above
	 * @return {const std::vector<geo_member>&} Ľ
	 *  Returns the results according the searching conditions.
	 */
	const std::vector<geo_member>& georadius(const char* key,
		double longitude, double latitude, double radius,
		int unit = GEO_UNIT_M,
		int with = GEO_WITH_COORD | GEO_WITH_DIST,
		int sort = GEO_SORT_ASC);

	/**
	 * þĳָλڸ뷶Χڵ
	 * Query a sorted set representing a geospatial index to fetch
	 * members matching a given maximum distance from a member
	 * @param key {const char*} Ӧļֵ
	 *  the specified key
	 * @param member {const char*} ĳָԱ
	 *  the specified member of a geospatial index
	 * @param radius {double} ޶ľ뷶ΧС
	 *  the distance from the specified coordinate
	 * @param unit {int} radius ĵλ
	 *  the unit type of the raidus
	 * @param with {int} ѯѡμĶ壺GEO_WITH_XXX
	 *  the serach operations, defined as GEO_WITH_XXX above
	 * @param sort {int} ѯʽμGEO_SORT_XXX
	 *  the sorted type of the results, defined as GEO_SORT_XXX above
	 * @return {const std::vector<geo_member>&} Ľ
	 *  Returns the results according the searching conditions.
	 */
	const std::vector<geo_member>& georadiusbymember(const char* key,
		const char* member, double radius,
		int unit = GEO_UNIT_M,
		int with = GEO_WITH_COORD | GEO_WITH_DIST,
		int sort = GEO_SORT_ASC);

private:
	std::vector<geo_member> positions_;

	void add_one_pos(const redis_result& rr);
	static const char* get_unit(int unit);
};

} // namespace acl

#endif // !defined(ACL_CLIENT_ONLY) && !defined(ACL_REDIS_DISABLE)
