// ------------------------------------------------------------------------
// 版权信息
// 版权归百小僧及百签科技（广东）有限公司所有。
// 所有权利保留。
// 官方网站：https://baiqian.com
//
// 许可证信息
// Furion 项目主要遵循 MIT 许可证和 Apache 许可证（版本 2.0）进行分发和使用。
// 许可证的完整文本可以在源代码树根目录中的 LICENSE-APACHE 和 LICENSE-MIT 文件中找到。
// 官方网站：https://furion.net
//
// 使用条款
// 使用本代码应遵守相关法律法规和许可证的要求。
//
// 免责声明
// 对于因使用本代码而产生的任何直接、间接、偶然、特殊或后果性损害，我们不承担任何责任。
//
// 其他重要信息
// Furion 项目的版权、商标、专利和其他相关权利均受相应法律法规的保护。
// 有关 Furion 项目的其他详细信息，请参阅位于源代码树根目录中的 COPYRIGHT 和 DISCLAIMER 文件。
//
// 更多信息
// 请访问 https://gitee.com/dotnetchina/Furion 获取更多关于 Furion 项目的许可证和版权信息。
// ------------------------------------------------------------------------

using System.Collections;

namespace Furion.Tools.CommandLine;

/// <summary>
/// Cli 输出消息
/// </summary>
public static partial class Cli
{
    /// <summary>
    /// 输出空行
    /// </summary>
    public static void EmptyLine()
    {
        Console.WriteLine();
    }

    /// <summary>
    /// 输出一行
    /// </summary>
    /// <param name="message"></param>
    /// <param name="color"></param>
    /// <param name="backgroundColor"></param>
    /// <param name="fillLine"></param>
    public static void WriteLine(object message, ConsoleColor color = ConsoleColor.White, ConsoleColor? backgroundColor = null, bool fillLine = false)
    {
        Output(message, text => Console.WriteLine(text), color, backgroundColor, fillLine);
    }

    /// <summary>
    /// 输出单个
    /// </summary>
    /// <param name="message"></param>
    /// <param name="color"></param>
    /// <param name="backgroundColor"></param>
    /// <param name="fillLine"></param>
    public static void Write(object message, ConsoleColor color = ConsoleColor.White, ConsoleColor? backgroundColor = null, bool fillLine = false)
    {
        Output(message, text => Console.Write(text), color, backgroundColor, fillLine);
    }

    /// <summary>
    /// 命令输出警告
    /// </summary>
    /// <param name="message"></param>
    public static void Warn(object message)
    {
        WriteLine(message, ConsoleColor.Yellow);
    }

    /// <summary>
    /// 命令输出成功提示一行
    /// </summary>
    /// <param name="message"></param>
    public static void Success(object message)
    {
        WriteLine(message, ConsoleColor.Green);
    }

    /// <summary>
    /// 命令输出错误
    /// </summary>
    /// <param name="message"></param>
    public static void Error(object message)
    {
        WriteLine(message, ConsoleColor.Red);
    }

    /// <summary>
    /// 命令输出提示
    /// </summary>
    /// <param name="message"></param>
    public static void Tip(object message)
    {
        WriteLine(message, ConsoleColor.Blue);
    }

    /// <summary>
    /// 收集用户多行输入
    /// </summary>
    /// <returns></returns>
    public static string[] ReadInput(string tip = "Enter 'exit' to exit.")
    {
        var list = new List<string>();

        // 输出一行提示
        Warn(tip);

        // 循环读取用户输入
        while (true)
        {
            Console.Write("> ");

            // 接收用户输入
            var input = Console.ReadLine();

            // 如果用户输入 exit 则退出
            if (input == "exit") break;

            list.Add(input);
        }

        return list.ToArray();
    }

    /// <summary>
    /// 读取用户选项
    /// </summary>
    /// <param name="options"></param>
    /// <returns></returns>
    public static int ReadOptions(string text, string[] options, string tip = "Enter 'exit' to exit.")
    {
        if (options == null || options.Length == 0) throw new ArgumentNullException(nameof(options));

        // 输出问题
        Repeat: Success(text);

        // 输出选项
        for (var i = 0; i < options.Length; i++)
        {
            Tip($" {i + 1}.{options[i]}");
        }

        // 输出一行提示
        Warn(tip);

        Console.Write("> ");

        // 接收用户输入
        var input = Console.ReadLine();

        // 如果用户输入 exit 则退出
        if (input == "exit") return -1;

        // 解析用户选择的选项
        var isNumber = int.TryParse(input, out var number);
        if (isNumber && number > 0 && number <= options.Length) return number;
        else goto Repeat;
    }

    /// <summary>
    /// 获取帮助
    /// </summary>
    /// <param name="keyword"></param>
    public static void GetHelpText(string keyword = default)
    {
        if (!string.IsNullOrWhiteSpace(keyword))
        {
            Warn("USAGE:");
            Console.WriteLine($"    {keyword} [OPTIONS]");
            Console.WriteLine();
        }

        Warn("OPTIONS:");

        foreach (var item in ArgumentMetadatas)
        {
            Console.WriteLine($"    -{item.ShortName}, --{(item.LongName + " <" + item.Property.PropertyType.Name.ToLower() + ">"),-30} {item.HelpText}");
        }
    }

    /// <summary>
    /// 命令行输出文本
    /// </summary>
    /// <param name="message">消息</param>
    /// <param name="handler"></param>
    /// <param name="color">字体颜色</param>
    /// <param name="backgroundColor">背景颜色</param>
    /// <param name="fillLine">背景颜色填充一整行</param>
    private static void Output(object message, Action<string> handler, ConsoleColor color = ConsoleColor.White, ConsoleColor? backgroundColor = null, bool fillLine = false)
    {
        Console.ResetColor();
        Console.ForegroundColor = color;
        if (backgroundColor != null) Console.BackgroundColor = backgroundColor.Value;

        var text = string.Empty;
        if (message != null)
        {
            var messageType = message.GetType();
            // 如果是基元类型，直接转换字符串输出
            if (messageType.IsPrimitive) text = message.ToString();
            // 如果是数组类型，直接转换字符串输出
            else if (messageType.IsArray) text = string.Join(",", ((IList)message).Cast<string>());
            // 如果是枚举类型
            else if (messageType.IsEnum) text = Enum.GetName(messageType, message);
            else text = message.ToString();
        }

        handler(!fillLine ? text : text.PadRight(Console.WindowWidth - text.Length));
        Console.ResetColor();
    }
}