package application

import (
	"context"
	"mayfly-go/internal/tag/domain/entity"
	"mayfly-go/internal/tag/domain/repository"
	"mayfly-go/pkg/base"
	"mayfly-go/pkg/contextx"
	"mayfly-go/pkg/errorx"
	"mayfly-go/pkg/model"
	"time"
)

type ResourceOpLog interface {
	base.App[*entity.ResourceOpLog]

	// AddResourceOpLog 新增资源操作记录
	AddResourceOpLog(ctx context.Context, codePath string) error
}

type resourceOpLogAppImpl struct {
	base.AppImpl[*entity.ResourceOpLog, repository.ResourceOpLog]

	tagTreeApp TagTree `inject:"TagTreeApp"`
}

var _ (ResourceOpLog) = (*resourceOpLogAppImpl)(nil)

// 注入ResourceOpLogRepo
func (rol *resourceOpLogAppImpl) InjectResourceOpLogRepo(resourceOpLogRepo repository.ResourceOpLog) {
	rol.Repo = resourceOpLogRepo
}

func (rol *resourceOpLogAppImpl) AddResourceOpLog(ctx context.Context, codePath string) error {
	loginAccount := contextx.GetLoginAccount(ctx)
	if loginAccount == nil {
		return errorx.NewBiz("Login information does not exist in this context")
	}

	var logs []*entity.ResourceOpLog
	qc := model.NewCond().Ge("create_time", time.Now().Add(-5*time.Minute)).Eq("creator_id", loginAccount.Id).Eq("code_path", codePath)
	logs, err := rol.ListByCond(qc)
	if err != nil {
		return err
	}
	// 指定时间内多次操作则不记录
	if len(logs) > 0 {
		return nil
	}
	tagTree := &entity.TagTree{CodePath: codePath}
	if err := rol.tagTreeApp.GetByCond(tagTree); err != nil {
		return errorx.NewBiz("resource not found")
	}

	return rol.Save(ctx, &entity.ResourceOpLog{
		ResourceCode: tagTree.Code,
		ResourceType: int8(tagTree.Type),
		CodePath:     tagTree.CodePath,
	})
}
